// word.h - Definitions for words and wordsets
/*begin-discard*/

// This file is part of the program spar , which is part of the PhD
// dissertation of Steve Lindblad, University of Nebraska, Lincoln,
// December 2003.

/*end-discard*/

#ifndef WORD_H
#define WORD_H

#include <string>
#include <set>
#include <list>
#include <vector>
#include <map>
#include <algorithm>
#include <iostream>
#include <iomanip>
#include <fstream>
#include <cstdlib>
#include <cctype> // for toupper and tolower
#include <cstdio> // for sprintf

/*begin-discard*/
// "Infinity" definition
// Here's the preferable way, if it works:
//#include <limits>
//static const int INFTY=numeric_limits<int>::max();
// Otherwise, try the old ugly C way:
/*end-discard*/
#include <limits.h>
static const int INFTY=INT_MAX;


/*
** Miscellaneous
*/

char *itoa (int n);
string reducestr (string x);
inline char cinv (const char x) {
    return (islower(x))?toupper(x):tolower(x); }
inline int mod (const int a, const int n)
   { return (n==0)?a:((a<0)?(((a%n)+n)%n):(a%n)); }


/*
** Words and Sets of Words
*/

class pointedpiece;


class Word {
    
  public:
    Word (const string& x="1"); // the constructor; default value 1
    Word operator+ (const Word& y) const // free monoid product
        { return Word(w+y.w); }
    Word operator* (const Word& y) const // free group product
        { return Word(reducestr(w+y.w)); }
    bool operator< (const Word& y) const;
    bool operator== (const Word& y) const {return w==y.w;}
    Word reduced() const {return Word(reducestr(w));}
    string str() const {return w;}
    int len() const {return w.size();}
    Word substr(int pos, int len) const;
    Word right(int pos) const { return substr(pos,len()-pos); }
    Word inv() const { return substr(0,-len()); }
    bool iscycred() const;
    bool issparse() const;
    bool check_sparse_with_violators 
        (pointedpiece& olx, pointedpiece& oly) const;
    // check_sparse_with_violators() assumes w is cyclically reduced.
    friend ostream& operator<< (ostream& s, const Word& x);
    friend ostream& operator<< (ostream& s, const set<Word>& A);
    
  private:
    string w;
};

extern const Word Word1; // empty word


/*begin-tex
\labelcoderef{coderef:pointedpiece}
end-tex*/
class pointedpiece {
  public:
    // The pointed piece has associated segment (i,w[i,m],+1)
    // and home segment (j,w[j,eta*m],eta) .
    Word w; // word for which this is a pointed piece.
    int i; // starting position of associated segment
    int j; // starting position of home segment
    int eta; // direction (+/-1) of home segment
    int m; // length of segments (>0)
    pointedpiece(Word u=Word1, int ti=0, int tj=0, int teta=0, int tm=0)
        { w=u; i=mod(ti,w.len()); j=mod(tj,w.len()); eta=teta; m=tm; }
    bool operator< (const pointedpiece& y) const {
        if (w<y.w) return true; else if (w>y.w) return false;
        if (i<y.i) return true; else if (i>y.i) return false;
        if (j<y.j) return true; else if (j>y.j) return false;
        if (m<y.m) return true; else if (m>y.m) return false;
        if (eta<y.eta) return true; else return false;
    }
    bool operator== (const pointedpiece& y) const {
        return w==y.w && i==y.i && j==y.j && m==y.m && eta==y.eta; }
    bool overlaps (const pointedpiece& y) const;
    friend ostream& operator<< (ostream& s, const pointedpiece& x);
};

#endif // WORD_H
