// word.cc - Words and sets of words
/*begin-discard*/

// This file is part of the program spar , which is part of the PhD
// dissertation of Steve Lindblad, University of Nebraska, Lincoln,
// December 2003.

/*end-discard*/

#include "word.h"


Word::Word (const string& x)
{
  if (x=="1" || x=="$" || x=="word1" || x=="eps" || x=="epsilon")
    w="";
  else
    w=x;
}


const Word Word1=Word(""); // empty word


bool Word::operator< (const Word& y) const
{
  int i,n=w.length(),s=n-y.w.length();
  if (s<0)
    return true;
  if (s>0)
    return false;
  const char *a=w.c_str(),*b=y.w.c_str();
  const char *ap,*bp;
  for (ap=a, bp=b; *ap==*bp && *ap; ++ap,++bp)
      ;
  if (!*ap)
    return false; // strings are equal
  if (toupper(*ap)==toupper(*bp))
    return *ap>*bp; // I want lower<upper, ASCII is lower>upper.
  else
    return toupper(*ap)<toupper(*bp);
}



/*begin-tex
\labelcoderef{coderef:substr}
end-tex*/
Word Word::substr(int pos, int len) const
{
    int n=w.size();
    string s="";
    if (n>0 && len!=0) {
        if (len>0)
            while (len--)
                s+=w[mod(pos++,n)];
        else
            while (len++)
                s+=cinv(w[mod(--pos,n)]);
    }
    return Word(s);
}


bool Word::iscycred() const
{
    return len()==0 || (substr(0,-1)!=substr(0,+1) && w==reducestr(w));
}


bool inzone(int k, int i, int m, int n)
// Returns whether k is in {i,i+1,...,i+m} if m>0;
//                      or {i,i-1,...,i-|m|} if m<0,
// where the values are mod n.
{
    if (m>0) {
        int d=mod(k-i,n);
        if (0<=d && d<=m) return true;
        else return false;
    }
    else { // m<=0
        int d=mod(i-k,n);
        if (0<=d && d<=-m) return true;
        else return false;
    }
}


bool zonesintersect(int i, int mi, int j, int mj, int n)
// Returns whether or not z(i,mi) intersects z(j,mj) based
// on a word length of n.  The zones intersect if the beginning
// or end of either is in the zone of the other.
{
    if (inzone(i   ,j,mj,n)) return true;
    if (inzone(i+mi,j,mj,n)) return true;
    if (inzone(j   ,i,mi,n)) return true;
    if (inzone(j+mj,i,mi,n)) return true;
    return false;
}


bool pointedpiece::overlaps (const pointedpiece& y) const
{
    // This function determines whether the pointed pieces overlap.
    // Think of *this as pointed piece (q,r) in the definition of
    // overlap and y as pointed piece (s,t).
    
    // z(q) and z(t) intersect
    if (zonesintersect(i,  m,  y.j,y.eta*y.m,w.len()))
        return true;

    // z(s) and z(r) intersect
    if (zonesintersect(y.i,y.m,  j,eta*m,    w.len()))
        return true;

    // z(q) and z(s) intersect and (q,r) and (s,t) aren't the same
    if (*this==y) return false;
    if (zonesintersect(i,m,y.i,y.m,w.len()))
        return true;

    return false;
}


/*begin-tex
\labelcoderef{coderef:checksparse}
end-tex*/
bool Word::check_sparse_with_violators
   (pointedpiece& olx, pointedpiece& oly) const
{
    // This function assumes w is cyclically reduced and of length at
    // least 2.  If w is not sparse, this function returns false and
    // olx and oly are pointed pieces that overlap.  If w is sparse,
    // this function returns true and the values of olx and oly are
    // undefined.

    set<pointedpiece> pp; // set of pointed pieces found so far
    
    for (int k=1; k<w.size(); ++k) {
        int eta;
        if (substr(k,+1)==substr(0,+1)||substr(k,-1)==substr(0,-1))
            eta=+1;
        else if (substr(k,+1)==substr(0,-1)||substr(k,-1)==substr(0,+1))
            eta=-1;
        else
            continue; // no pointed piece here; continue with next k

        int fl=0; // length of forward matching
        while (substr(k+fl,+1)==substr(0+eta*fl,eta) && fl<len())
            ++fl;
        int bl=0; // length of backword matching
        if (fl<len()) {
            while (substr(k-bl,-1)==substr(0-eta*bl,-eta))
                ++bl;
        }
        else {
            // Word is a proper power.  The forward matching shows
            // this (by matching the full length of the word), so just
            // let bl be 0.  The pointed piece is going to be found to
            // overlap itself in the next step.
        }

        olx=pointedpiece(*this,k-bl,0-eta*bl,eta,bl+fl);
        // We explicitly check for olx overlapping itself rather than
        // just putting olx in pp and allowing the check to occur
        // inside the loop because if olx does overlap itself, we
        // prefer to report that as being the sparseness violation.
        if (olx.overlaps(olx)) {
            oly=olx;
            return false;
        }
        for (set<pointedpiece>::const_iterator
                 p=pp.begin(); p!=pp.end(); ++p) {
            if (olx.overlaps(*p)) {
                oly=*p;
                return false;
            }
        }
        pp.insert(olx);
    }

    return true;
}


bool Word::issparse() const
{
    if (len()<2)
        return false;
    if (!iscycred())
        return false;
    pointedpiece olx,oly;
    return check_sparse_with_violators(olx,oly);
}



/*
** Miscellaneous
*/

char *itoa (int n)
{
  static char buf[100];
  sprintf(buf,"%d",n);
  return buf;
}

string reducestr (string x) // reduce word x (cancel inverses)
{
  string::size_type i=0;
  while (i+1<x.length()) {
    if (x[i]==cinv(x[i+1])) {
      x.erase(i,2);
      if (i>0) --i;
    }
    else 
      ++i;
  }
  return x;
}



/*
** Output routines
*/

ostream& operator<< (ostream& s, const Word& x)
{
  return s << (x.w==""?"epsilon":x.w);
}


ostream& operator<< (ostream& s, const set<Word>& A)
{
  int n=0;
  s << "{";
  for (set<Word>::iterator i=A.begin(); i!=A.end(); ++i,++n) {
    if (n>0) s << ',';
    s << *i;
  }
  s << "}";
  return s;
}


ostream& operator<< (ostream& s, const pointedpiece& x)
{
    return s << "((" 
             << x.i << "," 
             << x.w.substr(x.i,x.m) << ","
             << "1),("
             << x.j << ","
             << x.w.substr(x.j,x.eta*x.m) << ","
             << x.eta << "))";
}
