// spar.cc - Analyze the SC(1) for M=Inv<X|w=1> for w sparse.
/*begin-discard*/

// This file is part of the program spar , which is part of the PhD
// dissertation of Steve Lindblad, University of Nebraska, Lincoln,
// December 2003.

/*end-discard*/

#include "word.h"
#include "sc.h"

/*begin-discard*/

/*
** Exceptions
*/

class TooFew {public: TooFew(const string& t="") {s=t;} string s;};
class BadCmd {public: BadCmd(const string& t="") {s=t;} string s;};
class BadArg {public: BadArg(const string& t="") {s=t;} string s;};




/*
** Operand list
*/

class OperListEntry {

  public:
    OperListEntry (const int& x) { type=INT; n=x; }
    OperListEntry (const Word& x) { type=WORD; w=x; }
    OperListEntry (      InvMonoid* x) { type=INVMONOID; Mp=x; }
    OperListEntry (const InstDesc& x) { type=INSTDESC; id=x; }
    
  public:
    enum { INT, WORD, INVMONOID, INSTDESC } type;
    int n;
    Word w;
    InvMonoid *Mp;
    InstDesc id;
};

typedef vector<OperListEntry> OperList;




/*
** State
*/

class State;

class CmdEntry {
  public:
    char* name;
    char* (*exec)(const char *cmd, State& st);
};

class State {
  public:
    OperList ol;
    int execlevel; // level of nested exec files
    int execverbosity; /* verb. level for exec files:
                          0:none;1:show start/stop */
    int argc; 
    char **argv; // program arguments passed from main()
    CmdEntry *op; // array of commands
};


void docmd (const char *cmd, State& st)
{
  int k;

  // try the command as-is first
  for (k=0; st.op[k].name!=0 && strcasecmp(cmd,st.op[k].name)!=0; ++k)
      ;
  if (st.op[k].name!=0) {
      st.op[k].exec(cmd,st);
      return;
  }

  // if cmd begins with '-', try without the '-'
  if (cmd[0]=='-' && !isdigit(cmd[1])) {
      for (k=0; st.op[k].name!=0 && strcasecmp(cmd+1,st.op[k].name)!=0;
           ++k)
          ;
      if (st.op[k].name!=0) {
          st.op[k].exec(cmd,st);
          return;
      }
      else
          throw BadCmd("no such command '"+string(cmd)+"'");
  }

  // otherwise it's a word or number
  if (atoi(cmd)!=0 || strcmp(cmd,"0")==0)
      st.ol.push_back(atoi(cmd));
  else
      st.ol.push_back(Word(cmd));
}



void executefile (const char *fn, State& st)
{
  ifstream f(fn);
  if (!f) throw BadArg("can't open "+string(fn));
  string s;
  char c;

  ++st.execlevel;
  if (st.execverbosity==1 || st.execverbosity==3) {
    cout << "--- Starting execution of " 
         << fn << "(" << st.execlevel << ") ---" << endl;
  }

  while (f) {
    while (f>>c && c=='#')
      getline(f,s);  
    if (f) {
      f.putback(c);
      f >> s;
      docmd(s.c_str(),st);
    }
  }
  if (st.execverbosity==1 || st.execverbosity==3) {
    cout << "--- Execution of " 
         << fn << "(" << st.execlevel << ") complete ---" << endl;
  }
  --st.execlevel;
}

        
void help (const char *fn, State& st)
{
    cout << "Commands for spar" << endl;
    cout << "Usage: " << fn << " operands/commands" << endl;
    cout << 
"Operands are stored on an operand list (or stack).  Specify your commands
in postfix (RPN) form.  For example, use 'spar abcdef 3 -6 substr' to
compute w[3,-6] where w=abcdef." << endl;

    for (int k=0; st.op[k].name!=0; ++k) {
        string help=st.op[k].exec(0,st);
        string usage,description;
        int i=help.find("|");
        if (i==string::npos) {
            usage=help;
            description="";
        }
        else {
            usage=help.substr(0,i);
            description=help.substr(i+1,string::npos);
        }
        cout << setw(14) << setiosflags(ios::left) << st.op[k].name
             << description << ": " << usage << endl;
    }
}



/*
** Operations (to be assigned to commands)
*/

char* op_dispargs (const char *cmd, State& st)
{
    if (cmd==0) return
                    "-->|display argument with which program was called";
    cout << "Spar was run with the following command line: " << endl;
    for (int i=0; i<st.argc; ++i)
        cout << st.argv[i] << " ";
    cout << endl;
}



char* op_drop (const char *cmd, State& st)
{
    if (cmd==0) return
                    "x -->|delete x from operand list";
    if (st.ol.size()<1) throw TooFew(cmd);
    st.ol.pop_back();
}



char* op_dup (const char *cmd, State& st)
{
    if (cmd==0) return
                    "x --> x x|duplicate bottom of operand list";
    if (st.ol.size()<1) throw TooFew(cmd);
    OperListEntry x=st.ol.back();
    st.ol.push_back(x);
}



char* op_exec (const char *cmd, State& st)
{
    if (cmd==0) return
      "file -->|execute commands from file (comment lines beg w/#)";
    if (st.ol.size()<1)
        throw TooFew(string(cmd)+": give file name to execute");
    OperListEntry x=st.ol.back(); st.ol.pop_back();
    if (x.type!=OperListEntry::WORD)
        throw BadArg(string(cmd)+" 1:file name");
    executefile(x.w.str().c_str(),st);
}



char* op_execverbosity (const char *cmd, State& st)
{
    if (cmd==0) return
      "n -->|exec file indicators: 1=indicate begin/end, 0=don't";
    if (st.ol.size()<1)
        throw TooFew(string(cmd)+": give verbosity level");
    OperListEntry x=st.ol.back(); st.ol.pop_back();
    if (x.type!=OperListEntry::INT)
        throw BadArg(string(cmd)+" 1:verb. level");
    st.execverbosity=x.n;
}



char* op_geofsa (const char *cmd, State& st)
/* Displays the geodesic automaton for word. */
{
    if (cmd==0) return
      "M --> |display geodesic (conetype) automaton for 1";
    if (st.ol.size()<1) throw TooFew(cmd);
    OperListEntry x=st.ol.back(); st.ol.pop_back();
    if (x.type!=OperListEntry::INVMONOID) 
        throw BadArg(string(cmd)+" 1:invmonoid");
    cout << ConeTypeFSA(x.Mp->sc1p) << endl;
}



char* op_idadvance (const char *cmd, State& st)
{
    if (cmd==0) return
                    "id --> id|advance ID as much as possible";
    if (st.ol.size()<1) throw TooFew(cmd);
    OperListEntry x=st.ol.back(); st.ol.pop_back();
    if (x.type!=OperListEntry::INSTDESC) 
        throw BadArg(string(cmd)+" 1:instdesc");
    st.ol.push_back(x.id.advancedto());
}



char* op_idequal (const char *cmd, State& st)
/* op_idequal determines whether two stack items are equal, when only
the remaining portions of their words are considered.  This can be
used, for example, to test whether two words in the R-class of 1 are
equal.  Just use op_idadvance to advance the ID of each as much as
possible.  If both ID's are at the end of their words, and they are
equal by idequal, then the words are equal. */
{
    if (cmd==0) return
                    "y x --> 0/1|whether IDs are equal (ignoring past)";
    if (st.ol.size()<2) throw TooFew(cmd);
    OperListEntry x=st.ol.back(); st.ol.pop_back();
    if (x.type!=OperListEntry::INSTDESC) 
        throw BadArg(string(cmd)+" 1:instdesc");
    OperListEntry y=st.ol.back(); st.ol.pop_back();
    if (y.type!=OperListEntry::INSTDESC) 
        throw BadArg(string(cmd)+" 2:instdesc");
    st.ol.push_back(x.id.futequal(y.id));
}



char* op_idstep (const char *cmd, State& st)
{
    if (cmd==0) return
                    "id n --> id|advance ID at most n steps";
    if (st.ol.size()<2) throw TooFew(cmd);
    OperListEntry x=st.ol.back(); st.ol.pop_back();
    if (x.type!=OperListEntry::INT) 
        throw BadArg(string(cmd)+" 1:count");
    OperListEntry y=st.ol.back(); st.ol.pop_back();
    if (y.type!=OperListEntry::INSTDESC) 
        throw BadArg(string(cmd)+" 2:instdesc");
    x.n=max(0,x.n);
    st.ol.push_back(y.id.advancedto(0,0,x.n));
}



char* op_instdesc (const char *cmd, State& st)
{
    if (cmd==0) return
      "M t --> ID(M,t)|initial ID for reading t in SC(1)";
    if (st.ol.size()<2) throw TooFew(cmd);
    OperListEntry x=st.ol.back(); st.ol.pop_back();
    if (x.type!=OperListEntry::WORD) 
        throw BadArg(string(cmd)+" 1:word");
    OperListEntry y=st.ol.back(); st.ol.pop_back();
    if (y.type!=OperListEntry::INVMONOID) 
        throw BadArg(string(cmd)+" 2:invmonoid");
    st.ol.push_back(InstDesc(y.Mp->sc1p,x.w,0));
}



char* op_invmonoid (const char *cmd, State& st)
{
    if (cmd==0) return
                    "w --> M|inverse monoid Inv<X|w=1>";
    if (st.ol.size()<1) throw TooFew(cmd);
    OperListEntry x=st.ol.back(); st.ol.pop_back();
    if (x.type!=OperListEntry::WORD || !x.w.issparse())
        throw BadArg(string(cmd)+" 1:sparse word");
    st.ol.push_back(new InvMonoid(x.w));
}



char* op_issparse (const char *cmd, State& st)
{
    if (cmd==0) return
                    "w --> 0/1|whether or not w is sparse";
    if (st.ol.size()<1) throw TooFew(cmd);
    OperListEntry x=st.ol.back(); st.ol.pop_back();
    if (x.type!=OperListEntry::WORD) 
        throw BadArg(cmd);
    st.ol.push_back(x.w.issparse());
}



char* op_ndrop (const char *cmd, State& st)
{
    if (cmd==0) return
      "xn .. x1 n -->|delete bottom n operands from list";
    if (st.ol.size()<1) throw TooFew(cmd);
    OperListEntry x=st.ol.back(); st.ol.pop_back();
    if (x.type!=OperListEntry::INT) throw BadArg(cmd);
    while (st.ol.size()>0 && x.n--)
        st.ol.pop_back();
}



char* op_pick (const char *cmd, State& st)
{
    if (cmd==0) return
      "xn .. x1 n --> xn .. x1 xn|copy the nth item to bottom";
    if (st.ol.size()<1) throw TooFew(cmd);
    OperListEntry x=st.ol.back(); st.ol.pop_back();
    if (x.type!=OperListEntry::INT) throw BadArg(cmd);
    int n=x.n;
    if (st.ol.size()<n) throw TooFew(cmd);
    if (n>0) {
        x=st.ol[st.ol.size()-n];
        st.ol.push_back(x);
    }
}



char* op_over (const char *cmd, State& st)
{
    if (cmd==0) return
      "y x --> y x y|copy y to bottom of list";
    st.ol.push_back(OperListEntry(2));
    op_pick(cmd,st);
}



char* op_plus (const char *cmd, State& st)
{
    if (cmd==0) return
      "y x --> y+x|free monoid prod (concat) of words or sum of ints";
    if (st.ol.size()<2) throw TooFew(cmd);
    OperListEntry x=st.ol.back(); st.ol.pop_back();
    OperListEntry y=st.ol.back(); st.ol.pop_back();
    if (x.type==OperListEntry::INT && y.type==OperListEntry::INT)
        st.ol.push_back(y.n+x.n);
    else if (x.type==OperListEntry::WORD && y.type==OperListEntry::WORD)
        st.ol.push_back(y.w+x.w);
    else
        throw BadArg(string(cmd));
}



char* op_roll (const char *cmd, State& st)
{
    if (cmd==0) return
      "xn .. x1 n --> xn-1 .. x1 xn|roll n items of list up";
    if (st.ol.size()<1) throw TooFew(cmd);
    OperListEntry x=st.ol.back(); st.ol.pop_back();
    if (x.type!=OperListEntry::INT) throw BadArg(cmd);
    int n=x.n;
    if (st.ol.size()<n) throw TooFew(cmd);
    if (n>0) {
        x=st.ol[st.ol.size()-n];
        st.ol.erase(&st.ol[st.ol.size()-n]);
        st.ol.push_back(x);
    }
}



char* op_showitem (const char *cmd, State& st)
{
    if (cmd==0) return
                    "x -->|display details of item x";
    if (st.ol.size()<1) throw TooFew(cmd);
    OperListEntry x=st.ol.back(); st.ol.pop_back();
    switch (x.type) {
    case OperListEntry::INT:        cout << x.n; break;
    case OperListEntry::WORD:       cout << x.w;
        if (x.w==Word1) cout << " (empty word)"; break;
    case OperListEntry::INVMONOID:  cout << *(x.Mp); break;
    case OperListEntry::INSTDESC:   cout << x.id; break;
    }
    cout << endl;
}



char* op_showlist (const char *cmd, State& st)
{
    if (cmd==0) return
                    "-->|display entire operand list";
    int n=st.ol.size();
    if (n==0) {
        cout << "Empty operand list" << endl;
    }
    else {
        for (OperList::iterator
                 i=st.ol.begin(); i!=st.ol.end(); ++i, --n) {
            cout << n << ": ";
            switch (i->type) {
            case OperListEntry::INT:
                cout << i->n; break;
            case OperListEntry::INVMONOID:  
                cout << "Inv<X|"<<i->Mp->w<<"=1>"; break;
            case OperListEntry::INSTDESC:
                cout << i->id; break;
            case OperListEntry::WORD:
                cout << i->w;
                if (i->w==Word1) cout << " (empty word)"; break;
            }
            cout << endl;
        }
    }
}



char* op_substr (const char *cmd, State& st)
{
    if (cmd==0) return
                    "w i l --> w[i,l]|substring";
    if (st.ol.size()<3) throw TooFew(cmd);
    OperListEntry stl=st.ol.back(); st.ol.pop_back();
    OperListEntry sti=st.ol.back(); st.ol.pop_back();
    OperListEntry stw=st.ol.back(); st.ol.pop_back();
    if (stw.type!=OperListEntry::WORD || sti.type!=OperListEntry::INT
        || stl.type!=OperListEntry::INT) 
        throw BadArg(string(cmd)+" (w i l)");
    st.ol.push_back(stw.w.substr(sti.n,stl.n));
}



char* op_swap (const char *cmd, State& st)
{
    if (cmd==0) return
      "y x --> x y|swap bottom two items on operand list";
    st.ol.push_back(OperListEntry(2));
    op_roll(cmd,st);
}



char* op_times (const char *cmd, State& st)
{
    if (cmd==0) return
      "y x --> yx|free group prod (reduced) of words or prod of ints";
    if (st.ol.size()<2) throw TooFew(cmd);
    OperListEntry x=st.ol.back(); st.ol.pop_back();
    OperListEntry y=st.ol.back(); st.ol.pop_back();
    if (x.type==OperListEntry::INT && y.type==OperListEntry::INT)
        st.ol.push_back(y.n*x.n);
    else if (x.type==OperListEntry::WORD && y.type==OperListEntry::WORD)
        st.ol.push_back(y.w*x.w);
    else
        throw BadArg(string(cmd));
}



char* op_unroll (const char *cmd, State& st)
{
    if (cmd==0) return
      "xn .. x1 n --> x1 xn .. xn-1|roll n items of list down";
    if (st.ol.size()<1) throw TooFew(cmd);
    OperListEntry x=st.ol.back(); st.ol.pop_back();
    if (x.type!=OperListEntry::INT) throw BadArg(cmd);
    int n=x.n;
    if (st.ol.size()<n) throw TooFew(cmd);
    if (n>0) {
        x=st.ol.back();
        st.ol.insert(&st.ol[st.ol.size()-n],x);
        st.ol.pop_back();
    }
}



/*
** Array of command names
*/

CmdEntry op[] = {
     "*",             &op_times,
     "+",             &op_plus,
     "-go",           &op_idadvance,
     "-id",           &op_instdesc,
     "-invm",         &op_invmonoid,
     "-vx",           &op_execverbosity,
     "-x",            &op_exec,
     "advance",       &op_idadvance,
     "dispargs",      &op_dispargs,
     "drop",          &op_drop,
     "dup",           &op_dup,
     "exec",          &op_exec,
     "execverbosity", &op_execverbosity,
     "geofsa",        &op_geofsa,
     "idequal",       &op_idequal,
     "instdesc",      &op_instdesc,
     "invmonoid",     &op_invmonoid,
     "issparse",      &op_issparse,
     "list",          &op_showlist,
     "ndrop",         &op_ndrop,
     "over",          &op_over,
     "pick",          &op_pick,
     "plus",          &op_plus,
     "roll",          &op_roll,
     "rolld",         &op_unroll,
     "show",          &op_showitem,
     "showitem",      &op_showitem,
     "showlist",      &op_showlist,
     "step",          &op_idstep,
     "substr",        &op_substr,
     "swap",          &op_swap,
     "times",         &op_times,
     "unroll",        &op_unroll,
     0,               0
};

State st;



/*end-discard*/
/*
** Interactive mode
*/

string wordhelp=
"Input a word, such as abABcdCD.  Use upper-/lowercase letters for the
inverses of their lower-/uppercase counterparts (a^-1=A, A^-1=a, etc.).
You may enter 1, eps, epsilon, or nothing to specify the empty word.";

string stephelp=
"Enter an integer number of steps to perform (letters to read) before
I stop and ask you again.  The process automatically stops when the
entire word has been read or can be read no further, so you may enter
a large number such as 999 to continue without stopping.  You may
also simply press ENTER to do just one more step.";

string menuhelp=
"Choose from among the following options:
  1) List the PDA (face types and transitions) for your inverse monoid
  2) Interactively test whether or not u=1; i.e., whether SC(1) accepts u
  3) Test whether u,v are R-related to 1, and if so, whether u=v
  4) Compute the cone type automaton which recognizes geodesics in SC(1)
  8) Enter a new relator w (your current monoid is discarded)
  9) Leave this program (quit and Ctrl-D also work)";

string ask(const string& prompt, const string& help)
{
    string input;
    bool notdone=1;
    while (notdone) {
        cout << prompt;
        if (!getline(cin,input) || strcasecmp(input.c_str(),"quit")==0) {
            cout << "Good-bye!" << endl;
            exit(0);
        }
        if (input=="?" || strcasecmp(input.c_str(),"help")==0) {
            cout << help << endl;
        }
        else
            notdone=0;
    }
    return input;
}


void interactive_u_equals_v(InvMonoid& M)
{
    cout<< "\n"
        << "Enter words u and v.  This will determine whether or not\n"
        << "u and v are R-related to 1, and if so, whether or not u=v:"
        << endl;
    Word u=ask("u=",wordhelp);
    Word v=ask("v=",wordhelp);
    InstDesc idu=InstDesc(M.sc1p,u).advancedto();
    InstDesc idv=InstDesc(M.sc1p,v).advancedto();
    bool uR1=(idu.i==idu.u.len());
    bool vR1=(idv.i==idv.u.len());
    cout<< "Your word u " << (uR1?"IS":"is NOT") 
        << " R-related to 1." << endl;
    cout<< "Your word v " << (vR1?"IS":"is NOT") 
        << " R-related to 1." << endl;
    if (uR1 && vR1) {
        if (idu.qp==idv.qp && idu.stk==idv.stk)
            cout<< "The words u and v are EQUAL in M." << endl;
        else
            cout<< "The words u and v are EQUAL in M." << endl;
    }
    cout<< endl;
}


void interactive_step_id_display(InstDesc id)
{
    cout<< "Step " << id.i << ":"
        << " State=" << id.qp
        << "  Remaining=" << id.u.right(id.i)
        << "  Stack:";
    for (list<Vertex *>::const_iterator
             i=id.stk.begin(); i!=id.stk.end(); ++i)
        cout << " " << *i;
    cout<< endl;
}

void interactive_accepts_u_step(InvMonoid& M)
{
    cout<< "\n"
        << "Enter the word u which you wish to try to read in SC(1).\n"
        << "You will see the attempt to read u in SC(1) step-by-step."
        << endl;
    Word u=ask("u=",wordhelp);
    cout<< endl;

    InstDesc id=InstDesc(M.sc1p,u);
    bool notdone=1;
    interactive_step_id_display(id);

    while (notdone) {
        string nstep=ask("Next (or num of steps): ",stephelp);
        int n;
        if (nstep=="")
            n=1;
        else
            n=atoi(nstep.c_str());
        while (n-- && notdone) {
            /*begin-tex
            \labelcoderef{coderef:optstep}
            end-tex*/
            InstDesc newid=id.advancedto(0,0,1);
            notdone=(id.i<newid.i) && (newid.i<newid.u.len());
            id=newid;
            interactive_step_id_display(id);
        }
    }

    cout<< endl;
    if (id.i<id.u.len()) {
        cout<< "The first " << id.i 
            << " letters of u can be read in SC(1),\n"
            << "but not all of u.  Therefore, u is not R-related to 1." 
            << endl;
    }
    else if (id.qp!=id.scp->bcp->termp) {
        cout<< "All of u can be read inside SC(1), but the final state\n"
            << "attained is not the accept state, " 
            << id.scp->bcp->termp << ".\n"
            << "Therefore, u is NOT equal to 1.  However, this does mean\n"
            << "u is equivalent to 1 under Green's R-relation."
            << endl;
    }
    else {
        cout << "SC(1) accepts u. Therefore, u = 1." << endl;
    }
    cout<< endl;
}


void interactive_list_M(InvMonoid& M)
{
    cout << endl << "Your monoid is M=" << M;
}


void interactive_conetypefsa(InvMonoid& M)
{
    cout<< "\n"
        << "The following is the cone type automaton.  It recognizes\n"
        << "the language of geodesic words in the Schutzenberger\n"
        << "graph of 1:\n"
        << endl;
    cout << ConeTypeFSA(M.sc1p) << endl;
}


void interactive_mode(void)
{
    cout<<"
Welcome to 'spar'.  This program computes the PDA associated with the
Schutzenberger complex of 1 for inverse monoids of the form
M=Inv<X|w=1>, where w is a sparse word, and allows you to test whether
words are accepted by this PDA.

(Note: When this program asks for input, you may respond with ? or
'help' to get more information on what kind of response is required.
Also, you may run this program by giving operands and commands on
the command line and in files; type 'spar help' at the command line
for more information.)
";

    bool notdone=1;
    while (notdone) {
        Word w=ask("\nEnter your relator w: ",wordhelp);

        if (w.len()<2) {
            cout<< "Your relator, " << w << ", has length " 
                << w.len() << ".\n"
                << "Please enter a sparse relator of length "
                << "at least 2.\n"
                << "If you wish to stop this program, enter  quit ."
                << endl;
            continue;
        }

        if (!w.iscycred()) {
            cout<< "Your relator, " << w
                << ", is not cyclically reduced.\n"
                << "Please enter a cyclically reduced and sparse "
                << "relator." << endl;
            continue;
        }

        pointedpiece olx,oly;
        if (!w.check_sparse_with_violators(olx,oly)) {
            if (olx==oly) {
                cout<< "Your relator, " << w 
                    << ", is not sparse because the pointed piece\n"
                    << olx << " overlaps itself." << endl;
            }
            else {
                cout<< "Your relator, " << w 
                    << ", is not sparse because the pointed pieces\n"
                    << olx << " and " << oly << " overlap." << endl;
            }
            cout<< "Please enter a sparse relator." << endl;
            continue;
        }

        InvMonoid M=InvMonoid(w);
        cout<< "\nI have computed the Schutzenberger complex of 1 for\n"
            << "M=Inv<X|w=1>, w=" << w << "." << endl;
        cout<< endl << menuhelp << endl;
        string menuprompt=
            "What next (1:list,2:u=1,3:u=v,4:geo,8:newrel,9:quit)? ";
        string option;
        while ((option=ask(menuprompt,menuhelp))!="9" && option!="8") {
                 if (option=="1") interactive_list_M(M);
            else if (option=="2") interactive_accepts_u_step(M);
            else if (option=="3") interactive_u_equals_v(M);
            else if (option=="4") interactive_conetypefsa(M);
            else cout << endl << menuhelp << endl;
        }
        notdone=(option=="8");
    }

    cout << "Good-bye!" << endl;
    exit(0);
}


int main (int argc, char **argv)
{
    if (argc==1)
        interactive_mode();
    /*begin-discard*/

    st.argc=argc; 
    st.argv=argv;
    st.execverbosity=0;
    st.execlevel=0;
    st.op=op;

    if (argc==2 && (strcasecmp(argv[1],"help")==0
                    || strcasecmp(argv[1],"--help")==0)) {
        help(argv[0],st);
        exit(0);
    }

    try {executefile("sparconf.spar",st);}
    catch (BadArg e) {
        // just ignore any problems reading the config file
    }

    for (int i=1; i<argc; ++i) {
        try {
            docmd(argv[i],st);
        }
        catch (TooFew e) { 
            cout << "Too few arguments (" << e.s << ")" << endl;
            exit(1);
        }
        catch (BadCmd e) {
            cout << "Bad command: " << e.s << endl;
            exit(1);
        }
        catch (BadArg e) {
            cout << "Bad argument: " << e.s << endl;
            exit(1);
        }
    }
  
    // print final operand list
    if (st.ol.size()>1) {
        cout << endl << "Final operand list:" << endl;
        op_showlist("-list",st);
    }
    else if (st.ol.size()>0) {
        op_showitem("-showitem",st);
    }
    /*end-discard*/
}
