// sc.h - Definitions for Schutzenberger complexes
/*begin-discard*/

// This file is part of the program spar , which is part of the PhD
// dissertation of Steve Lindblad, University of Nebraska, Lincoln,
// December 2003.

/*end-discard*/

#ifndef SC_H
#define SC_H

#include "word.h"

const int fieldwidth=13;

class InvMonoid;
class SchComplex;
class FaceType;
class Face;
class Vertex;
class TransDom;
class TransRan;
class InstDesc;


class TransDom { // domain of transition function
  public:
    Word x;    // letter labeling edge
    Vertex *t; // top stack letter
               // (0 indicates it can be anything---i.e., "t")
    TransDom(Word y=Word1, Vertex *s=0) { x=y; t=s; }
    bool operator< (const TransDom& y) const {
        if (t<y.t) return true; else if (t>y.t) return false;
        if (x<y.x) return true; else            return false;
    }
    bool operator== (const TransDom& y) const {
        return (t==y.t) && (x==y.x); }
    friend ostream& operator<< (ostream& s, const TransDom& td);
};


class TransRan { // range of transition function
  public:
    int dir; // direction (-1:toward, 0:neither, +1:away from origin)
    Vertex *destp; // destination vertex of edge
    enum TransStackOp { NONE, PUSH, POP } stkop;
    Vertex *pushed; // vertex to push if stkop==PUSH
    TransRan(int d=0, Vertex *dp=0, TransStackOp so=NONE, Vertex *vp=0) {
        dir=d; destp=dp; stkop=so; pushed=vp; }
    bool operator== (const TransRan& y) const {
        return (dir==y.dir) && (destp==y.destp) && (stkop==y.stkop)
            && (pushed==y.pushed); }
    friend ostream& operator<< (ostream& s, const TransRan& tr);
};

extern const TransRan NullTransRan;


class FaceType {
  public:
    InvMonoid* Mp; // Mp==0 indicates empty (dummy) facetype
    enum { BASECOMPLEX, FACE } mode;
    Word baseword; // BASECOMPLEX: ==Word1, word used to build complex
    int b,f;   // FACE: last index folded when reading fwd or back
    int sink2; // FACE: twice the index of geodesic sink

    FaceType() { Mp=0; } // EmptyFaceType
    FaceType (InvMonoid* Mptr, Word t);
    FaceType (InvMonoid* Mptr, int j=0, int i=0, int k=0);
    int bvmod(int j) const; // 1..n
    int fvmod(int i) const; // 0..n-1
    int bv(void) const { return bvmod(b); }
    int fv(void) const { return fvmod(f); }
    bool operator< (const FaceType& y) const {
        if (Mp<y.Mp) return true; else if (Mp>y.Mp) return false;
        if (mode<y.mode) return true; else if (mode>y.mode) return false;
        if (mode==BASECOMPLEX) return baseword<y.baseword;
        if (f<y.f)         return true; else if (f>y.f) return false;
        if (b<y.b)         return true; else if (b>y.b) return false;
        if (sink2<y.sink2) return true; else            return false;
    }
    bool operator== (const FaceType& y) const {
        if (Mp!=y.Mp) return false;
        if (mode!=y.mode) return false;
        if (mode==BASECOMPLEX) return baseword==y.baseword;
        return (f==y.f) && (b==y.b) && (sink2==y.sink2); 
    }
    friend ostream& operator<< (ostream& s, const FaceType& ct);
};

extern const FaceType EmptyFaceType;


/*begin-tex
\labelcoderef{coderef:vertex}
end-tex*/
class Vertex {
 public:
    Face *ownerp; // face to which vertex belongs
    int index; // index of vertex inside its face (the vp[] array)
    FaceType attachedfacetype; // type of face attached at this vertex
    map<TransDom,TransRan> edge; // edges to other vertices
    Vertex(Face *owner=0, int i=0)  { 
        ownerp=owner; index=i; attachedfacetype=EmptyFaceType; }
    void fixgeodesics(void);
    void maxfoldparams(int* bip, Vertex* *bpp, int* fip, Vertex* *fpp,
                       int* ep);
    void attachface(void);
    TransRan findtrans (TransDom td);
    friend string vertexstr (const Vertex* vp);
    friend ostream& operator<< (ostream& s, const Vertex* vp);
    friend ostream& operator<< (ostream& s, const Vertex& v);
};


/*begin-tex
\labelcoderef{coderef:face}
end-tex*/
class Face {
 public:
    FaceType facetype;
    vector<Vertex *> vp; // ptrs to vertices of face
    // facetype.mode==FACE: indexed facetype.fv()+1..facetype.bv()-1.
    // facetype.mode==BASECOMPLEX: index 0 only

    // for facetype.mode==BASECOMPLEX:
    Vertex *initp; // initial vertex of base complex
    Vertex *termp; // terminal vertex of base complex
    
    Face (FaceType ftyp);
    void buildfacebottom (int f, int b, int sink2);
    void BuildBaseComplex(void);
    friend ostream& operator<< (ostream& s, const Face& C);
};


class SchComplex {
 public:
    InvMonoid *Mp; // inverse monoid to which this SchComplex belongs
    Face *bcp; // underlying base complex
    SchComplex(InvMonoid* Mptr, Word t);
    bool accepts(Word t);
    InstDesc advance (InstDesc id, Vertex *term, int plus=0);
    friend ostream& operator<< (ostream& s, const SchComplex& sc);
};


/*begin-tex
\labelcoderef{coderef:invmonoid}
end-tex*/
class InvMonoid { // =Inv<X|w=1>
 public:
    Word w; // relator for inverse monoid
    int n; // word len
    SchComplex *sc1p; // SchComplex of 1
    map<FaceType, Face *> ftyptbl; // all known face types
    InvMonoid() { n=0; sc1p=0; } // dummy InvMonoid
    InvMonoid (Word u);
    Word trim_identity_words (Word t);
    bool sc1accepts (Word u) { return sc1p->accepts(u); }
    friend ostream& operator<< (ostream& s, const InvMonoid& M);
};





/*begin-tex
\labelcoderef{coderef:instdesc}
end-tex*/
class InstDesc { // instantenous description
  public:
    SchComplex *scp;
    Vertex *qp; // current state in PDA
    Word u;     // word being read
    int i;      // index to current position in word
    list<Vertex *> stk; // stack
    InstDesc() { scp=0; }
    InstDesc(SchComplex *sp, Word t=Word1, int j=0);
    InstDesc advancedto (Vertex *term=0,
                         int plus=0, int maxcount=-1) const;
    bool futequal (const InstDesc& y) const;
    friend ostream& operator<< (ostream& s, const InstDesc& id);
};


class ConeTypeFSA {
  public:
    SchComplex *scp; // SchComplex on which this FSA is based
    map<Vertex *,int> iofv;
    vector<Vertex *> vofi;
    vector<int> iofi;
    int nstate;
    set<Word> alphabet;
    ConeTypeFSA(SchComplex *sp);
    int dest(int vi, Word x);
    friend ostream& operator<< (ostream& s, const ConeTypeFSA& fsa);
};


// Constants
extern const FaceType EmptyFaceType;
extern const TransRan NullTransRan;
extern Vertex DummyVertex;
extern Vertex *const DummyStackLetter;

#endif // SC_H
