// sc.cc - Build Schutzenberger complex
/*begin-discard*/

// This file is part of the program spar , which is part of the PhD
// dissertation of Steve Lindblad, University of Nebraska, Lincoln,
// December 2003.

/*end-discard*/

#include "sc.h"
#include <sstream>


InvMonoid::InvMonoid (Word u)
{
    w=u;
    n=w.len();
    sc1p=new SchComplex(this,Word1);
}


SchComplex::SchComplex (InvMonoid *Mptr, Word t)
{
    /* assumes t=Word1 */
    Mp=Mptr;
    bcp=new Face(FaceType(Mp,t));
    bcp->vp[0]->attachface(); // build SC(1)
    Mp->ftyptbl[bcp->facetype]=bcp;
}


Face::Face (FaceType ftyp)
// This function creates the vertices of a face and creates all
// transitions between vertices within the face.  No push/pop
// transitions between faces are created; they are created in
// SchComplex() or attachface().
{
    facetype=ftyp;
    /*begin-tex
    \labelcoderef{coderef:mainconstr}
    end-tex*/
    if (ftyp.mode==FaceType::BASECOMPLEX) {
        vp.reserve(1);
        initp=termp=new Vertex(this,0);
        vp.push_back(initp); // set vp[0]=initp=termp
    }
    else { // FACE
        vp.reserve(facetype.Mp->w.len());
        buildfacebottom(facetype.bv(),facetype.fv(),facetype.sink2);
    }
}


FaceType::FaceType (InvMonoid* Mptr, Word t)
{
    mode=BASECOMPLEX;
    Mp=Mptr;
    baseword=Mp->trim_identity_words(t);
}


FaceType::FaceType (InvMonoid* Mptr, int j=0, int i=0, int k=0)
{
    mode=FACE;
    Mp=Mptr;
    b=fvmod(j);
    f=fvmod(i);
    sink2=k;
}


int FaceType::bvmod(int j) const { return mod(j-1,Mp->n)+1; } // 1..n
int FaceType::fvmod(int i) const { return mod(i,  Mp->n)  ; } // 0..n-1

/*begin-tex
\labelcoderef{coderef:buildbot}
end-tex*/
void Face::buildfacebottom (int b, int f, int sink2)
// This routine creates the "bottom half," or unfolded portion, of a
// new face, added to the vp[] array of this face.  These are the new
// edges of the face, which do not fold onto its predecessor or the
// pre-existing edges of a base complex.
//
// f, b, sink2 = vertices with face indices f+1 .. b-1 are created,
// and the geodesic directions (on edges) are set so that the sink is
// at index sink2/2.
//
// Note that this function does not create any transitions from the
// vertices corresponding to face indices f and b.  That is the
// responsibility of the calling routine.
{
    vp.resize(b);
    // Vector vp[] now has entries vp[f+1],...,vp[b-1],
    // initialized to 0.  This is where pointers to the new vertices
    // will be stored:
    for (int i=f+1; i<=b-1; ++i)
        vp[i]=new Vertex(this,i);
    
    // Create edges going away from index f:
    for (int i=f+1; 2*(i+1)<=sink2; ++i) {
        Word x=facetype.Mp->w.substr(i,+1);
        vp[i]  ->edge[TransDom(x,0)]
            =TransRan(+1,vp[i+1],TransRan::NONE);
        vp[i+1]->edge[TransDom(x.inv(),0)]
            =TransRan(-1,vp[i]  ,TransRan::NONE);
    }
        
    // Create edges going away from index b:
    for (int j=b-1; 2*(j-1)>=sink2; --j) {
        Word x=facetype.Mp->w.substr(j,-1);
        vp[j]  ->edge[TransDom(x,0)]
            =TransRan(+1,vp[j-1],TransRan::NONE);
        vp[j-1]->edge[TransDom(x.inv(),0)]
            =TransRan(-1,vp[j]  ,TransRan::NONE);
    }

    // Create the last edge (with no geodesic direction), if the
    // sink lands in the middle of an edge:
    if (sink2%2) { 
        // if sink2 is odd, there's an edge with no direction
        
        int i=sink2/2;
        Word x=facetype.Mp->w.substr(i,+1);
        vp[i]  ->edge[TransDom(x,0)]
            =TransRan( 0,vp[i+1],TransRan::NONE);
        vp[i+1]->edge[TransDom(x.inv(),0)]
            =TransRan( 0,vp[i]  ,TransRan::NONE);
    }
}


/*begin-tex
\labelcoderef{coderef:attachface}
end-tex*/
void Vertex::attachface(void)
// This function determines what kind of face should be attached
// at the vertex, by looking forward and back to see how much folding
// can occur.  If a FACE of the required type exists, appropriate
// PUSH/POP transitions are added.  If a FACE of the required type
// does not exist, it is created and the PUSH/POP transitions are
// added.  If new vertices are created, attachface() is called
// recursively to attach faces at the new vertices.
{
    InvMonoid *Mp=ownerp->facetype.Mp;
    //int n=Mp->n;
    int b,f,e;
    Vertex *bp,*fp;
    maxfoldparams(&b,&bp,&f,&fp,&e);

    attachedfacetype=FaceType(Mp,b,f,b+f-e);
    // int r=j-i; // number of unfolded edges
    // e=d(O,fp)-d(O,bp)
    // kf=f+(r-e)/2, kb=b-(r+e)/2
    //   => k=(kb+kf)/2 => 2k=kb+kf=f+b-e = sink2
    
    // Get the desired face
    Face *ncp;
    if (Mp->ftyptbl.find(attachedfacetype)==Mp->ftyptbl.end()) {
        // FACE doesn't exist; create it.
        ncp=new Face(attachedfacetype);
        Mp->ftyptbl[attachedfacetype]=ncp;
        for (int i=f+1; i<=b-1; ++i)
            ncp->vp[i]->attachface();
    }
    else {
        ncp=Mp->ftyptbl[attachedfacetype];
    }
    
    // Create the between-face (pushing/popping) transitions
    Word x;
    Vertex *np;
    x=Mp->w.substr(f,+1);
    np=ncp->vp[f+1];
    // Stack "letter" (Vertex *)0 means that it doesn't matter what's
    // on the top of the stack.  This trick seems nicer than creating
    // transitions for all possible vertices.
    fp->edge[TransDom(x,0)]         =TransRan(+1,np,TransRan::PUSH,this);
    np->edge[TransDom(x.inv(),this)]=TransRan(-1,fp,TransRan::POP);
    x=Mp->w.substr(b,-1);
    np=ncp->vp[b-1];
    bp->edge[TransDom(x,0)]         =TransRan(+1,np,TransRan::PUSH,this);
    np->edge[TransDom(x.inv(),this)]=TransRan(-1,bp,TransRan::POP);
}



/*begin-tex
\labelcoderef{coderef:maxfoldparams}
end-tex*/
void Vertex::maxfoldparams(int* bip, Vertex* *bpp, int* fip,
                           Vertex* *fpp, int* ep)
// This function looks to see what happens if a face is attached at
// the current vertex and folded.  It returns b (in 1..n) and f (in
// 0..n-1) as the vertex indices (within the face being sewn on) at
// which the folding ends.  bp and bf are the corresponding Vertex *'s
// of the actual vertices (in the predecessor face).  e is the net
// count of geodesic directions traversed; in other words, e is
// d(O,fp)-d(O,bp), or how much closer b is to O (the origin of the
// Sch complex) than f.
{
    InvMonoid *Mp=ownerp->facetype.Mp;
    int n=Mp->n;
    int b=n; Vertex* bp=this;
    int f=0; Vertex* fp=this;
    int e=0; // d(O,fp)-d(O,bp), how much closer to O b is than f
    // e<0 => face folded back toward origin

    /* Look forward/backward for end of part onto which the new face
       folds.  Sparseness guarantees these loops will terminate. */
    map<TransDom,TransRan>::const_iterator dp;
    while ((dp=fp->edge.find(TransDom(Mp->w.substr(f,+1),0)))
           !=fp->edge.end()) {
        fp = dp->second.destp;
        e += dp->second.dir;
        ++f;
    }

    while ((dp=bp->edge.find(TransDom(Mp->w.substr(b,-1),0)))
           !=bp->edge.end()) {
        bp = dp->second.destp;
        e -= dp->second.dir;
        --b;
    }

    // store the answers
    *bip=b;
    *bpp=bp;
    *fip=f;
    *fpp=fp;
    *ep=e;
}



/*
** Routines to read within a PDA 
*/

InstDesc::InstDesc(SchComplex *sp, Word t, int j)
{
    scp=sp;
    qp=scp->bcp->initp;
    u=t;
    i=j;
    stk.push_front(DummyStackLetter); // initial letter on stack
}


bool InstDesc::futequal (const InstDesc& y) const
// Returns whether the two IDs are equal, ignoring the portions
// of the words already read.
{
    if (scp!=y.scp) return false;
    if (qp!=y.qp) return false; 
    if (u.right(i)!=y.u.right(y.i)) return false;
    if (stk!=y.stk) return false;
    return true;
}


TransRan Vertex::findtrans (TransDom td)
{
    map<TransDom,TransRan>::const_iterator ep;
    for (ep=edge.begin(); ep!=edge.end(); ++ep) {
        if (ep->first.x==td.x) {
            if (ep->first.t==0 || ep->first.t==td.t)
                return ep->second;
        }
    }
    return NullTransRan;
}


/*begin-tex
\labelcoderef{coderef:advancedto}
end-tex*/
InstDesc InstDesc::advancedto (Vertex *term=0, int plus=0,
                               int maxcount=-1) const
/* Reads id in the PDA *scp, until the end of the word is reached,
until Vertex *term is reached, or until maxcount transitions have been
made, whichever comes first.  If plus is set, then at least one edge
must be read before Vertex *term is attained.  A maxcount of -1 means
no maximum count.  The function returns the final InstDesc.
Advancedto() may be called with all arguments 0 (the default) to force
as much of the word as possible to be read.  Note that term is assumed
to be in the underlying Munn tree.  Then, the only way it can be
reached is when the stack is empty. */
{
    InstDesc id=*this;
    TransRan tr;
    int maxi;

    if (maxcount==-1)
        maxi=u.len(); // read to the end of the word
    else
        maxi=id.i+maxcount;
    while (
        (id.i<maxi)
        && ( id.qp!=term || (plus && id.i==i) )
        && (tr=id.qp->findtrans(
            TransDom(id.u.substr(id.i,1),id.stk.front()) ))!=NullTransRan
        ) {

        ++id.i;
        id.qp=tr.destp;
        if (tr.stkop==TransRan::PUSH)
            id.stk.push_front(tr.pushed);
        else if (tr.stkop==TransRan::POP)
            id.stk.pop_front();
    }
    return id;
}


Word InvMonoid::trim_identity_words (Word t)
// This routine requires sc1p be properly set for nonempty words.  It
// correctly returns the empty word if t is empty, regardless of
// whether sc1p has been defined or not.
{
    if (t.len()==0)
        return t;
    int i=0;
    while (i<t.len()) {
        InstDesc newid=InstDesc(sc1p,t,i).advancedto(sc1p->bcp->termp,1);
        if (i!=newid.i && newid.qp==sc1p->bcp->termp) {
            t=t.substr(0,i)+t.right(newid.i); // cut out part equal to 1
            // keep i at its current value to check next substr
        }
        else {
            ++i;
        }
    }
    return t;
}


bool SchComplex::accepts(Word t)
{
    InstDesc id=InstDesc(this,t,0).advancedto();
    return id.qp==bcp->termp && id.i==t.len();
    /* This word is accepted if all of it was read (id.i=t.len) AND
       reading stopped at the terminal vertex. */
}



/*
** Routines to produce the cone type automaton
*/

// The routines below implement the finite-state automaton
// minimization algorithm in Hopcroft and Ullman to produce the cone
// type automaton.  Note that the minimization algorithm assumes there
// is a transition for every letter of the alphabet.  Since this may
// not be case in the edge's, the implementation below makes use of an
// imaginary fail state, FAILSTATE.  All other states are accept
// states.

const int FAILSTATE=-1; // -1 indicates sink state (the only fail state)

class statepair {
public:
    int x,y;
    statepair(int a, int b) {
        if (a>b) { x=b; y=a; } else { x=a; y=b; }
    }
    bool operator< (const statepair& t) const {
        if (x<t.x) return true; else if (x>t.x) return false;
        if (y<t.y) return true; else return false;
    }
};

/*begin-tex
\labelcoderef{coderef:geofsadest}
end-tex*/
int ConeTypeFSA::dest(int vi, Word x)
{
    if (vi==FAILSTATE)
        return FAILSTATE;
    map<TransDom,TransRan>::const_iterator 
        dp=vofi[vi]->edge.find(TransDom(x,0));
    if (dp==vofi[vi]->edge.end() || dp->second.dir!=+1)
        return FAILSTATE;
    return iofv[dp->second.destp];
}


void FsaNumberChildren(Vertex* vp,  
   map<Vertex *,int>& iofv, vector<Vertex *>& vofi, set<Word>& alphabet)
{
    if (iofv.find(vp)==iofv.end()) {
        // haven't done this vertex yet, add to the list
        int n=vofi.size();
        iofv[vp]=n;
        vofi.push_back(vp); // add vofi[n] entry
        for (map<TransDom,TransRan>::const_iterator i=vp->edge.begin();
             i!=vp->edge.end(); i++) {
            alphabet.insert(i->first.x);
            if (i->second.dir==+1) {
                FsaNumberChildren(i->second.destp,iofv,vofi,alphabet);
            }
        }
    }
}


void markpair(statepair pq, set<statepair>& marked,
              map<statepair,set<statepair> >& listof)
{
    if (marked.find(pq)==marked.end()) {
        marked.insert(pq);
        for (set<statepair>::const_iterator sp=listof[pq].begin();
             sp!=listof[pq].end(); ++sp) {
            markpair(*sp,marked,listof);
        }
    }
}


/*begin-tex
\labelcoderef{coderef:geofsa}
end-tex*/
ConeTypeFSA::ConeTypeFSA(SchComplex *sp)
{
    scp=sp;
    FsaNumberChildren(scp->bcp->initp,iofv,vofi,alphabet);
    nstate=vofi.size();
    set<statepair> emptyset;
    map<statepair,set<statepair> > listof;
    set<statepair> marked;
    
    for (int p=0; p<nstate; ++p) 
        marked.insert(statepair(p,FAILSTATE));

    for (int p=-1; p<nstate; ++p) {
        for (int q=p+1; q<nstate; ++q) {
            listof[statepair(p,q)]=emptyset;
        }
    }

    for (int p=0; p<nstate; ++p) {
        for (int q=p+1; q<nstate; ++q) {
            int foundmarked=0;
            for (set<Word>::const_iterator xp=alphabet.begin();
                 (!foundmarked) && xp!=alphabet.end(); ++xp) {
                statepair destpair=statepair(dest(p,*xp),dest(q,*xp));
                if (marked.find(destpair)!=marked.end()) {
                    foundmarked=1;
                    markpair(statepair(p,q),marked,listof);
                }
            }

            if (!foundmarked) {
                for (set<Word>::const_iterator xp=alphabet.begin();
                     xp!=alphabet.end(); ++xp) {
                    statepair destpair=statepair(dest(p,*xp),dest(q,*xp));
                    if (destpair.x!=destpair.y) {
                        listof[destpair].insert(statepair(p,q));
                    }
                }
            }
        }
    }

    /*begin-tex
    \labelcoderef{coderef:geofsarenum}
    end-tex*/
    // Set iofi array so iofi[j] gives the new index of state j.
    for (int i=0; i<nstate; ++i)
        iofi.push_back(i);
    for (int i=0; i<nstate; ++i) {
        if (iofi[i]==i) { // this index hasn't changed
            for (int j=i+1; j<nstate; ++j) {
                statepair ij=statepair(i,j);
                if (marked.find(ij)==marked.end()) {
                    // i and j are the same
                    iofi[j]=i;
                }
            }
        }
    }
}



/*
** Output Routines
*/

/*begin-tex
\labelcoderef{coderef:geofsaoutput}
end-tex*/
ostream& operator<< (ostream& s, const ConeTypeFSA& fsa)
{
    int nconetype=0;
    for (int k=0; k<fsa.nstate; ++k) {
        Vertex *vp=fsa.vofi[k];
        if (fsa.iofi[k]==k) {
            ++nconetype;
        }
    }
    s << "Geodesic word acceptor for SC(1) ("
      << nconetype 
      << " cone types):" << endl;
    for (int k=0; k<fsa.nstate; ++k) {
        if (fsa.iofi[k]==k) {
            Vertex *vp=fsa.vofi[k];
            s << vp << ":";
            for (map<TransDom,TransRan>::const_iterator i=vp->edge.begin();
                 i!=vp->edge.end(); i++) {
                if (i->second.dir==+1) {
                    int j=fsa.iofi[fsa.iofv.find(i->second.destp)->second];
                    s << " " << i->first.x << "->" << fsa.vofi[j];
                }
            }
            s << endl;
        }
    }
    return s;
}
    
    
ostream& operator<< (ostream& s, const FaceType& bt)
{
    if (bt==EmptyFaceType) 
        return s << "(emptyfacetype)";
    else if (bt.mode==FaceType::BASECOMPLEX)
        return s << "(" << (bt.baseword) << ")";
    else return s
            << "("
            << (bt.b) << ","
            << (bt.f) << ","
            << (bt.sink2/2)
            << ((bt.sink2%2)?".5":"")
            << ")";
}



void VertexEdgeOut(ostream& s, Vertex& v)
{
    const char *arrow;
    s << "  Vertex " << v << ": attachedfacetype=" 
      << v.attachedfacetype << ", " 
      << v.edge.size() << " edges:" << endl;
    for (map<TransDom,TransRan>::const_iterator 
             i=v.edge.begin(); i!=v.edge.end(); i++) {
        if (i->second.dir>0)
            arrow="+1";
        else if (i->second.dir<0)
            arrow="-1";
        else
            arrow="0";
        s << "    ( " << i->first.x << " ,";
        if (i->first.t==0)
            s << setw(fieldwidth) << "t";
        else
            s << setw(fieldwidth) << vertexstr(i->first.t).c_str();
        // I need the c_str() to make setw(10) work properly.
        s << " )  ->  ("
          << setw(fieldwidth) << vertexstr(i->second.destp).c_str() 
          << " , ";
        if (i->second.stkop==TransRan::PUSH)
            s << setw(fieldwidth-2) 
              << vertexstr(i->second.pushed).c_str() 
              << " t";
        else if (i->second.stkop==TransRan::POP)
            s << setw(fieldwidth) << "eps";
        else
            s << setw(fieldwidth) << "t";
        s << " )" << setw(4) << arrow << endl;
    }
}


string vertexstr (const Vertex* vp)
{
    if (vp==0)
        return "-";
    else {
        ostringstream s;
        s << vp->ownerp->facetype << vp->index;
        return s.str();
    }
}


ostream& operator<< (ostream& s, const Vertex* vp)
{
    if (vp==0)
        return s << "-";
    else if (vp==DummyStackLetter)
        return s << "(stackstart)";
    else
        return s << *vp;
}


ostream& operator<< (ostream& s, const Vertex& v)
{
    return s << v.ownerp->facetype << v.index;
}


ostream& operator<< (ostream& s, const TransDom& td)
{
    s << "(" << td.x << "," << *td.t << ")";
    return s;
}


ostream& operator<< (ostream& s, const TransRan& tr)
{
    s << "(" 
      << tr.dir << "," 
      << tr.destp << "," 
      << tr.stkop << "," 
      << tr.pushed 
      << ",this=" << &tr << ")";
    return s;
}


ostream& operator<< (ostream& s, const Face& C)
{
    s << "FaceType" << C.facetype << ": ";
    if (C.facetype.mode==FaceType::BASECOMPLEX) {
        s << "BASECOMPLEX initial=" << C.initp
          << ", terminal=" << C.termp 
          << "; 1 vertex" << endl;
        VertexEdgeOut(s,*C.vp[0]);
    }
    else {
        s << "FACE  " 
          << (C.facetype.bv()-C.facetype.fv()-1) << " vertices"
          << endl;
        if (C.facetype.sink2==0) {
            VertexEdgeOut(s,*C.vp[0]);
        }
        else {
            for (int i=C.facetype.fv()+1; i<=C.facetype.bv()-1; ++i) {
                VertexEdgeOut(s,*C.vp[i]);
            }
        }
    }
    return s;
}


ostream& operator<< (ostream& s, const SchComplex& sc)
{
    s << "SchComplex for u=" << sc.bcp->facetype.baseword
      << " in M=Inv<X|w=" << sc.Mp->w << "=1>" << endl;
    s << "Base complex: " << *sc.bcp << endl;
    return s;
}


ostream& operator<< (ostream& s, const InvMonoid& M)
{
    s << "InvMonoid<X|w=" << M.w << "=1>  "
      << M.ftyptbl.size() << " face types:" << endl;
    for (map<FaceType,Face *>::const_iterator btap=M.ftyptbl.begin();
         btap!=M.ftyptbl.end(); btap++) {
        s << *(btap->second) << endl;
    }
    return s;
}


ostream& operator<< (ostream& s, const InstDesc& id)
{
    s << "ID("
      << id.scp->bcp->facetype.baseword << ","
      << id.u << ","
      << id.i << "; "
      << *(id.qp) << ", "
      << id.u.right(id.i) << ",";
    for (list<Vertex *>::const_iterator 
             i=id.stk.begin(); i!=id.stk.end(); ++i)
        s << " " << *i;
    s << ")";
    return s;
}



/*
** Miscellaneous definitions
*/

const FaceType EmptyFaceType;
const TransRan NullTransRan;
Vertex DummyVertex;
  /* don't make const--want to be able to pass DummyStackLetter to
     places where ptrs to non-const Vertex are allowed. */
Vertex *const DummyStackLetter=&DummyVertex;

