function [S,E,I,R,sinfty]=seir(R0,tL,tI,e0,i0,r0,target,maxdays)
%
% function [S,E,I,R,sinfty]=seir(R0,tL,tI,e0,i0,r0,target,maxdays)
%
% runs a simulation of an SEIR model
%
% S: susceptible
% E: 'exposed' ('latent' is the more accurate term)
% I: infectious
% R: removed
%
% R0 is the basic reproductive number
% tL is the mean incubation time
% tI is the mean recovery time
% e0 is the initial latent fraction
% i0 is the initial infectious fraction
% r0 is the initial immune fraction
% target is the infected fraction used as an end condition
% maxdays is the maximum simulation duration
%
% by Glenn Ledder
% written 2020/11/27
%
% direct comments to gledder@unl.edu

%% DATA

% suggested default values
% R0 = 5;
% tL = 2;
% tI = 10;
% e0 = 0.0001;
% i0 = 0;
% r0 = 0;
% target = 0.001;
% maxdays = 1000;

%% INITIALIZATION

% calculate parameters

eta = 1/tL;
gamma = 1/tI;
beta = R0*gamma;
s0 = 1-e0-i0-r0;
k = 1-r0-log(s0)/R0;

% set up results data structure with Y=[S,E,I,R]

results = zeros(maxdays+1,4);
Y = [s0,e0,i0,r0];
results(1,:) = Y;

y = Y';
oldx = e0+i0;

%% FUNCTION FOR Sinfty

fnc = @(s,R0,k) s-exp(R0*(s-k));    % parameterized function
fofs = @(s) fnc(s,R0,k);            % function of s alone
    
%% COMPUTATION

for t=1:maxdays
    % y is a column vector, Y^T
    y = rk4(1,y);
    Y = y';
    results(t+1,:) = Y;
    if Y(2)+Y(3)>min(target,oldx)
        oldx = Y(2)+Y(3);
    else
        results = results(1:(t+1),:);
        break;
    end
end

S = results(:,1);
E = results(:,2);
I = results(:,3);
R = results(:,4);

sinfty = fzero(fofs,[0,1]);

%% FUNCTION FOR rk4

    function y=rk4(dt,y0)
        % y0 is a column vector of initial conditions at t
        % y is a column vector of values at t+dt
        k1 = yprime(y0);
        k2 = yprime(y0+0.5*dt*k1);
        k3 = yprime(y0+0.5*dt*k2);
        k4 = yprime(y0+dt*k3);
        y = y0+dt*(k1+2*k2+2*k3+k4)/6;
    end
        
%% FUNCTION FOR THE DIFFERENTIAL EQUATION

    function yp=yprime(y)
    % split out components
        S = y(1);
        E = y(2);
        I = y(3);
    % compute derivatives
        Sp = -beta*S*I;
        Ep = -Sp-eta*E;
        Ip = eta*E-gamma*I;
        Rp = gamma*I;
    % assemble derivative
        yp = [Sp;Ep;Ip;Rp];
    end

%% END

end
