function y=TransitionSim()

% Plots and reports the results of a recovery simulation

%   assumes an Erlang Ek distribution

%   y is the number of people who are not fully recovered each day

%   mu is the mean recovery time
%   k is the number of equivalent phases
%   y0 is the initial count of particles
%   days is the length of the simulation

% by Glenn Ledder
% written 2020/12/30
% revised 2021/03/15

% direct comments to gledder@unl.edu

%% SCENARIO DATA

mu = 10;
k = 6;
y0 = 10000;
days = 30;

%% INITIALIZATION

% set up for figure

clf
hold on
box on

%% COMPUTATION

% collect transition data

y = transition(mu,k,y0,days);

%% OUTPUT

times = 0:days;

% use 'MarkerSize',6 in Octave
plot(times,y,'r.','MarkerSize',12)

% use 'FontSize',18 in Octave
xlabel('\it{t}','FontSize',14)
ylabel('sick','FontSize',14);

%% END

end


%% FUNCTION transition

function y=transition(mu,k,y0,days)
%
% function y=transition(mu,k,y0,days)
%
%   runs a simulation of disease recovery
%   assumes an Erlang Ek distribution
%
%   y is the number of people who are not fully recovered each day
%
%   mu is the mean recovery time
%   k is the number of equivalent phases
%   y0 is the initial count of particles
%   days is the length of the simulation

%% INITIALIZATION

% create a row vector

y = y0*ones(1,days+1);

%% COMPUTATION

% collect y0 values from Ek(t;mu)

times = randerlang(mu,k,y0);
times = sort(times);

% count undecayed particles for each day

for n=1:days
    y(n+1) = length(find(times>n));
end
    
%% END

end


%% FUNCTION randerlang(mu,k,n)

function [T,Tmean,Tstd]=randerlang(mu,k,n)

% produces a row vector T of n Erlang-distributed values
% also returns mean Tmean and standard deviation Tstd

% mu is the mean total time for the distribution
% k is the number of phases

% default n is 1

if nargin<3
    n = 1;
end

% collect uniformly distributed random values

Xexp = rand(k,n);

% convert values to match exponential distribution
T = -mu*log(Xexp)/k;
    
% add parts to get whole
if k>1
    T = sum(T);
end
Tmean = mean(T);
Tstd = std(T);

end

