%% SEIR parameter studies

% Runs a covid19 experiment to determine how maximum hospitalizations,
% total deaths, and final at risk percentage depend on a parameter value

% Uses seaiihrd.m, 2020/11/27 version

% User specifies a list of values for one of the key parameters:
%   R0 is the basic reproductive number
%   tL is the mean incubation time
%   tI is the mean recovery time
%   i0 is the initial infectious fraction
%   r0 is the initial immune fraction

% e0 is calculated assuming exponential growth phase

% The program is designed so that only a few lines need to be modified to
% make a new experiment (see '%%%' comments)
%   lines 49-51 define the independent variable values
%   line 54 identifies the x axis label for the graph
%   line 86 links the independent variable name and values

% Output figure:
%   top left panel: max infectious fraction
%   top right panel: day of max new infections
%   bottom left panel: fraction of susceptibles who become infected
%   bottom right panel: final fraction susceptibles

% by Glenn Ledder
% written 2020/11/27
% revised 2021/03/13

% direct comments to gledder@unl.edu

%% DEFAULT SCENARIO DATA

R0 = 2.5;
tL = 5;
tI = 10;
i0 = 0.001;
r0 = 0;

%% INDEPENDENT VARIABLE DATA

%%% This section needs to be modified for each experiment.

%%% first and last are the min and max values for the independent variable
%%% N is the number of points (not subdivisions)
first = .1;
last = 6;
N = 60;

%%% xname is the name for the x axis label
xname = 'R0';

%% COMMON DATA

target = 0.001;
maxdays = 1000;

%% INITIALIZATION

clf
for k=1:4
    subplot(2,2,k)
    hold on
    box on
    xlabel(xname)
end
colors = get(gca,'colororder');

% xvals holds whatever values are being used for the independent variable
xvals = linspace(first,last,N);

finalS = zeros(1,N);
fracI = zeros(1,N);
maxI = zeros(1,N);
maxday = zeros(1,N);

%% COMPUTATION

for n=1:N
    
    %%% The left side of this statement needs to be the independent
    %%% variable for the experiment.
    R0 = xvals(n);

    eta = 1/tL;
    gamma = 1/tI;
    beta = gamma*R0;

    a = eta;
    b = eta-gamma+gamma*R0*i0;
    c = -gamma*R0*(1-i0-r0);
    b2 = b/2;
    rho = (sqrt(b2^2-a*c)-b2)/a;
    e0 = rho*i0;
    s0 = 1-e0-i0-r0;

    [~,~,I,~,sinfty] = seir(R0,tL,tI,e0,i0,r0,target,maxdays);

    [M,J] = max(I);
    maxI(n) = M;
    maxday(n) = J-1;
    finalS(n) = sinfty;
    fracI(n) = (s0-sinfty)/s0;
end
    
%% OUTPUT

subplot(2,2,1)
plot(xvals,maxI,'Color',colors(3,:),'LineWidth',1.7)
ylabel('max fraction infectious')
subplot(2,2,2)
plot(xvals,maxday,'k','LineWidth',1.7)
ylabel('days for max infectious')
subplot(2,2,3)
plot(xvals,fracI,'Color',colors(2,:),'LineWidth',1.7)
ylabel('fraction infected')
subplot(2,2,4)
plot(xvals,finalS,'Color',colors(1,:),'LineWidth',1.7)
ylabel('final fraction susceptible')
