%% SEAIHRD parameter studies

% Runs a covid19 experiment to determine how maximum hospitalizations,
% total deaths, and final at risk percentage depend on a parameter value

% Uses seaihrd.m, 2022/03/06 version

% User specifies a list of values for one of the key parameters:
%   R0 is the basic reproductive number
%   c is the fraction of infectious patients (I+J) who are confirmed
%   delta is the contact factor
%   h0 is the initial hopitalization count per 100K

% The program is designed so that only a few lines need to be modified to
% make a new experiment (see '%%%' comments)
%   lines 48-50 define the independent variable values
%   line 53 identifies the x axis label for the graph
%   line 81 links the independent variable name and values

% Output figure:
%   top left panel: max hospitalizations (per 100K)
%       (compare to average of 280 hospital beds per 100K)
%   top right panel: thousands of US deaths (out of 325M)
%   bottom left panel: final susceptibles (percent)
%   bottom right panel: days for max hospitalizations and A+I+H<target

% by Glenn Ledder
% written 2022/01/25
% revised 2022/03/06

% direct comments to gledder@unl.edu

beds = 280;

%% DEFAULT SCENARIO DATA

R0 = 5.7;
c = 0.1;
delta = 1;
h0 = 1;

%% INDEPENDENT VARIABLE DATA

%%% This section needs to be modified for each experiment.

%%% first and last are the min and max values for the independent variable
%%% N is the number of points (not subdivisions)
first = 0;
last = 1;
N = 51;

%%% xname is the name for the x axis label
xname = 'delta';

%% INITIALIZATION

clf
for k=1:4
    subplot(2,2,k)
    hold on
    box on
    xlabel(xname)
end
colors = get(gca,'colororder');

% xvals holds whatever values are being used for the independent variable
xvals = linspace(first,last,N);

finalpctS = zeros(1,N);
maxHp100K = zeros(1,N);
USdeaths = zeros(1,N);
maxHtime = zeros(1,N);
endtime = zeros(1,N);

%% COMPUTATION

for n=1:N
    
    %%% The left side of this statement needs to be the independent
    %%% variable for the experiment.
    delta = xvals(n);

    [S,~,~,~,~,H,~,D] = seaihrd(R0,c,delta,h0);

    [maxHp100K(n),maxrow] = max(100000*H);
    maxHtime(n) = maxrow-1;

    USdeaths(n) = 325000*D(end);
    finalpctS(n) = 100*S(end);

    endtime(n) = length(S)-1;
end
    
%% OUTPUT

subplot(2,2,1)
plot(xvals,maxHp100K,'Color',colors(2,:),'LineWidth',1.7)
plot([first,last],[beds,beds],'k--')
subplot(2,2,2)
plot(xvals,USdeaths,'r','LineWidth',1.7)
subplot(2,2,3)
plot(xvals,finalpctS,'Color',colors(1,:),'LineWidth',1.7)
subplot(2,2,4)
plot(xvals,maxHtime,'Color',colors(2,:),'LineWidth',1.7)
plot(xvals,endtime,'Color',colors(1,:),'LineWidth',1.7)
legend('max hospitalized','end condition')

subplot(2,2,1)
ylabel('max hospitalized per 100K')
subplot(2,2,2)
ylabel('US deaths (1000s)')
subplot(2,2,3)
ylim([0,100])
ylabel('percent S at end')
subplot(2,2,4)
ylabel('days')

