function MMfit()

% Compares three linearizations with the semilinear method for model data

% Prints means for K, V, and Delta AIC

% by Glenn Ledder
% written 2021/01/15
% revised 2022/07/20

% direct comments to gledder@unl.edu

%% DATA

% Standard deviation of data adjustments
s = 0.02;

% Number of runs
% Decrease T to 2500 for free Octave online
T = 40000;

x = [0.2 0.4 0.7 1.0 1.4 2.0 2.8 3.8 5.0];

f = @(x,p) x./(p+x);

%% INITIALIZATION

% create exact data
Y = x./(1+x);

% create data structures
n = length(x);
K = ones(T,3);
V = ones(T,3);
K0 = ones(1,T);
V0 = ones(1,T);
RSS0 = zeros(T,1);
DeltaAIC = zeros(T,3);
better = zeros(1,3);
significant = zeros(1,3);
small = zeros(1,3);

muK = zeros(1,4);
muV = zeros(1,4);
muDeltaAIC = zeros(1,3);

%% COMPUTATION

for i=1:T
    % create trial data set
    r = 1+s*randn(1,n);
    y = r.*Y;
    
    % fit Lineweaver-Burk
    [m,b] = leastsq(1./x,1./y);
    V(i,1) = 1/b;
    K(i,1) = m/b;
    
    % fit Hanes-Woolf
    [m,b] = leastsq(x,x./y);
    V(i,2) = 1/m;
    K(i,2) = b/m;
    
    % fit Dowd-Riggs #3
    [m,b] = leastsq(y./x,y);
    V(i,3) = b;
    K(i,3) = -m;
    
    % fit semilinear
    [K0(i),V0(i),RSS0(i)] = semilinfit(f,0.98,1.02,0.001,x,y);

    % compute AIC differences
    for j=1:3
        v = V(i,j);
        k = K(i,j);
        ymod = v*x./(k+x);
        RSS = sum((y-ymod).^2);
        % count when results are "better" than semilinear
        if RSS<RSS0(i)
            better(j) = better(j)+1;
        end
            
        DeltaAIC(i,j) = n*log(RSS/RSS0(i));
        % count when DeltaAIC is significant or "small"
        if DeltaAIC(i,j)>=2
            significant(j) = significant(j)+1;
        elseif DeltaAIC(i,j)<0.5
            small(j) = small(j)+1;
        end
    end
    
end
   
% record outcomes
muK(4) = mean(K0);
muV(4) = mean(V0);
for j=1:3
    muK(j) = mean(K(:,j));
    muV(j) = mean(V(:,j));
    muDeltaAIC(j) = mean(DeltaAIC(:,j));
end

% convert to percentages
better = 100*better/T;
significant = 100*significant/T;
small = 100*small/T;

%% OUTPUT

muK
muV
muDeltaAIC

end


%% FUNCTION leastsq

function [m,b] = leastsq(x,y)
    xbar = sum(x)/length(x);
    ybar = sum(y)/length(y);
    X = x-xbar;
    Y = y-ybar;
    sumx2 = sum(X.^2);
    sumxy = sum(X.*Y);
    m = sumxy/sumx2;
    b = ybar-m*xbar;
end


%% FUNCTION semilinfit

function [pstar,Astar,RSS,AIC]=semilinfit(f,pmin,pmax,tol,x,y)

%   finds optimal (p,A) and RSS for the semilinear model y=Af(x;p)
%   f = @(x,p) must be defined in calling program

%   initial search in [pmin,pmax]
%   search interval shifts if no sign change in slope

%   tol is desired error tolerance for pstar

%   x,y are the lists of data values

% define function F(p) in terms of auxiliary function rss
sumy2 = sum(y.^2);
F = @(p) rss(f,p,x,y,sumy2);

% use auxiliary function findmin to find pstar
pstar = findmin(F,pmin,pmax,tol);

% find Astar, RSS, AIC
ff = f(x,pstar);
Astar = sum(y.*ff)/sum(ff.^2);
RSS = rss(f,pstar,x,y,sumy2);
n = length(x);
AIC = n*log(RSS/n)+6;

end %semilinfit


%% FUNCTION rss(f,p,x,y,sumy2)

function z=rss(f,p,x,y,sumy2)
%   computes RSS for the semilinear model y=qf(x;p)
%   x,y are the lists of data values
%   sumy2 is the sum of y^2 values
ff = f(x,p);
z = sumy2-(sum(y.*ff)).^2/sum(ff.^2);
end


%% FUNCTION findmin(f,xmin,xmax,tol)

function xstar=findmin(f,xmin,xmax,tol)

%   finds a local minimizer of a function f
%   designed for functions that cannot be vectorized
%   initial search in [xmin,xmax]
%   search interval shifts if no sign change in slope
%   does not work for functions with frequent sign changes in slope

fvals = zeros(1,11);
dx = 2*tol;

while dx>tol
    dx = 0.1*(xmax-xmin);
    xvals = xmin:dx:xmax;
    for i=1:length(xvals)
        fvals(i) = f(xvals(i));
    end
    [~,I] = min(fvals);
    if I==1
        xmax = xmin+dx;
        xmin = xmin-9*dx;
    elseif I==11
        xmin = xmax-dx;
        xmax = xmax+9*dx;
    else
        xstar = xvals(I);
        xmin = xstar-dx;
        xmax = xstar+dx;
    end
end %while

end %findmin
