%%% LynSys

% runs a simulation for a discrete linear system 
%   x_{t+1}=Mx_t of any size.
  
% The output includes estimates of the long-term growth rate 
%   and population structure.

% by Glenn Ledder
% written 2021/02/13
% revised 2021/03/15

% direct comments to gledder@unl.edu

%% DATA

% set model parameters

M = [0,104,160;0.01,0,0;0,0.3,0];

% set initial condition and total time

X0 = [1000 50 5];
T = 10;

% set scale factor for x1
%   population plot is x1/F rather than x1

F = 100;

% choose names for variable axes

varname1 = '\it{X}';
varname2 = '\it{X_t / X_{t-1}}';

% use 1 for square axes or 0 for nonfixed aspect ratio

sq = 0;

% set the top value for the y axis for the growth rate plot  
%   use 0 to accept the default

ymax = 0;

%% INITIALIZATION

% set up graphs for populations and growth rates

clf
for k=1:2
    subplot(1,2,k)
    hold on
    box on
    if sq>0
        axis square
    end
end

% set initial condition

X = X0';                % X is a column vector

% XX is a matrix with the population vectors as columns

sz = length(X0);
XX = zeros(T+1,sz);
XX(1,:) = X0;           % initial condition is row 0

%% COMPUTATION

% compute the long-term behavior

[V,D] = eig(M);
[evals,indices] = sort(diag(D));
Ds = D(indices,indices);
Vs = V(:,indices);
evec = abs(Vs(:,end));

% compute population values and update data structures

for t=1:T
    X = M*X;
    XX(t+1,:) = X';
end

% N is the vector of total populations

N = sum(XX')';

% L is the ratio L(j)=N(j+1)/N(j)

s = 1:T;
L = N(s+1)./N(s);

%% OUTPUT

% in Octave, use marker size of 6 for '.'

% plot populations

subplot(1,2,1)
plot(0:T,XX(:,1)/F,'.','MarkerSize',12)
for j=2:sz
    plot(0:T,XX(:,j),'.','MarkerSize',12)
end

% use 'FontSize',18 in Octave
xlabel('\it{t}','FontSize',14);
ylabel(varname1,'FontSize',14,'Rotation',0);

% plot growth rate

subplot(1,2,2)
plot(s,L,'.','MarkerSize',12)
if ymax~=0
    ylim([0 ymax]);
end

% use 'FontSize',18 in Octave
xlabel('\it{t}','FontSize',14);
% use 'FontSize',16 in Octave
ylabel(varname2,'FontSize',12);

% report long-term behavior

lambda = Ds(end)
smallest = min(evec);
proportions = evec/smallest
