%%% DiscreteVars

% runs a simulation for small discrete linear systems:

%   x1_(t+1)=m11 x1_t +m12 x2_t
%   x2_{t+1}=m21 x1_t +m22 x2_t

% or 

%   x1_(t+1)=m11 x1_t +m12 x2_t +m13 x3_t
%   x2_{t+1}=m21 x1_t +m22 x2_t +m23 x3_t
%   x3_{t+1}=m31 x1_t +m32 x2_t +m33 x3_t

% The output includes estimates of the long-term growth rates and
%   population ratios.

% by Glenn Ledder
% written 2021/02/13
% revised 2021/03/15

% direct comments to gledder@unl.edu

%% DATA

% set model parameters (mi3 and m3j ignored for size 2)

size = 3;
m11 = 0;
m12 = 104;
m13 = 160;
m21 = 0.01;
m22 = 0;
m23 = 0;
m31 = 0;
m32 = 0.3;
m33 = 0;

% set initial condition and total time

x10 = 1000;
x20 = 50;
x30 = 5;        % ignored if size = 2
T = 10;

% set scale factor for x1
%   population plot is x1/F rather than x1

F = 100;

% choose legend location

locn = 'northwest';

% use 1 for square axes or 0 for nonfixed aspect ratio

sq = 0;

% set the top value for the y axis for the growth rate plot  
%   use 0 to accept the default

ymax = 0;

% define new color(s) if desired

darkgreen = [0 0.6 0];

%% INITIALIZATION

% set up graphs for populations and growth rates

clf
for k=1:2
    subplot(1,2,k)
    hold on
    box on
    if sq>0
        axis square
    end
end

% Create data structures

x1 = x10*ones(1,T+1);
x2 = x20*ones(1,T+1);
x3 = x30*ones(1,T+1);   % ignored if size = 2

%% COMPUTATION

% compute population values

switch size
    case 2   
        for t=1:T
            x1(t+1) = m11*x1(t)+m12*x2(t);
            x2(t+1) = m21*x1(t)+m22*x2(t);
        end
    case 3
        for t=1:T
            x1(t+1) = m11*x1(t)+m12*x2(t)+m13*x3(t);
            x2(t+1) = m21*x1(t)+m22*x2(t)+m23*x3(t);
            x3(t+1) = m31*x1(t)+m32*x2(t)+m33*x3(t);
        end
end

% compute growth rates

s = 1:T;
y1 = x1(s+1)./x1(s);
y2 = x2(s+1)./x2(s);
if size==3
    y3 = x3(s+1)./x3(s);
end

%% OUTPUT

% in Octave, use marker sizes of
%   2.5 for '*'
%   3 for 's'
%   6 for '.'

% plot populations

subplot(1,2,1)
switch size
    case 2
        plot(0:T,x1/F,'b.','MarkerSize',12)
        plot(0:T,x2,'rs','MarkerFaceColor','r','MarkerSize',5)
    case 3
        plot(0:T,x1/F,'*','Color',darkgreen,'MarkerSize',5)
        plot(0:T,x2,'b.','MarkerSize',12)
        plot(0:T,x3,'rs','MarkerFaceColor','r','MarkerSize',5)
end

% use 'FontSize',18 in Octave
xlabel('\it{t}','FontSize',14);
ylabel('\it{X}','FontSize',14,'Rotation',0);

switch size
    case 2
        if F==1
            legend('x1','x2','Location',locn)
        else
            lbl = ['x1/' int2str(F)];
            legend(lbl,'x2','Location',locn)
        end
    case 3
        if F==1
            legend('x1','x2','x3','Location',locn)
        else
            lbl = ['x1/' int2str(F)];
            legend(lbl,'x2','x3','Location',locn)
        end
end

% plot growth rates

subplot(1,2,2)
switch size
    case 2
        plot(s,y1,'b.','MarkerSize',12)
        plot(s,y2,'rs','MarkerFaceColor','r','MarkerSize',5)
    case 3
        plot(s,y1,'*','Color',darkgreen,'MarkerSize',5)
        plot(s,y2,'b.','MarkerSize',12)
        plot(s,y3,'rs','MarkerFaceColor','r','MarkerSize',5)
end

if ymax~=0
    ylim([0 ymax]);
end

% use 'FontSize',18 in Octave
xlabel('\it{t}','FontSize',14);
% use 'FontSize',16 in Octave
ylabel('\it{X_t / X_{t-1}}','FontSize',12);

% report last values of growth rates and population ratios

switch size
    case 2   
        growthrates = [y1(end) y2(end)]
        ratios = x1(end)/x2(end)
    case 3
        growthrates = [y1(end) y2(end) y3(end)]
        ratios = [x1(end) x2(end)]/x3(end)
end
