function [S,E,A,I,J,H,R,D]=seaihrd(R0,c,delta,h0)
%
% function [S,E,A,I,J,H,R,D]=seaihrd(R0,c,delta,h0)
%
% runs a simulation of an SEAIHRD model
%
% S: susceptible
% E: 'exposed' ('latent' is the more accurate term)
% A: asymptomatic
% I: infectious, not requiring hospitalization (I1)
% J: pre-hospitalized (I2)
% H: hospitalized
% R: recovered
% D: deceased
%
% R0 is the basic reproductive number
% pc is the fraction of infectious patients (I+J) who are confirmed
% delta is the contact factor
% h0 is the initial hopitalization count per 100K

% by Glenn Ledder
% written 2022/01/23
% revised 2022/03/06

% direct comments to gledder@unl.edu

%% DATA

% suggested default values
% R0 = 5.7;
% c = 0.1;
% delta = 0.25;
% h0 = 1

fa = 0.75;
fc = 0.1;
m = 0.25;
pa = 0.4;
ph = 0.018;
te = 5;
ta = 8;
ti = 10;
tj = 3;
th = 10;

target = 0.001;
maxdays = 1000;

h0 = h0/100000;

%% INITIALIZATION

% calculate parameters

eta = 1/te;
alpha = 1/ta;
gamma = 1/ti;
sigma = 1/tj;
nu = 1/th;

T = fa*pa*ta+(1-pa-ph)*ti+ph*tj;
beta = R0/T;
lambda = fzero(@(L) betaoflambda(L,eta,alpha,gamma,sigma,pa,ph,fa)-beta,0.2);

% calculate initial conditions

a = pa*eta./(lambda+alpha);
i = (1-pa-ph)*eta./(lambda+gamma);
j = ph*eta./(lambda+sigma);
h = sigma*j./(lambda+nu);
e0 = h0/h;
a0 = a*e0;
i0 = i*e0;
j0 = j*e0;
s0 = 1-e0-a0-i0-j0-h0;

% set up results data structure with Y=[S,E,A,I,J,H,R,D]

results = zeros(maxdays+1,8);
Y = [s0,e0,a0,i0,j0,h0,0,0];
results(1,:) = Y;

y = Y';
oldx = e0+i0+j0+a0;
     
%% COMPUTATION

for t=1:maxdays
    % y is a column vector, Y^T
    y = rk4(1,y);
    Y = y';
    results(t+1,:) = Y;
    if sum(Y(2:5))>min(target,oldx)
        oldx = sum(Y(2:5));
    else
        results = results(1:(t+1),:);
        break;
    end
end

S = results(:,1);
E = results(:,2);
A = results(:,3);
I = results(:,4);
J = results(:,5);
H = results(:,6);
R = results(:,7);
D = results(:,8);

%% FUNCTION FOR rk4

    function y=rk4(dt,y0)
        % y0 is a column vector of initial conditions at t
        % y is a column vector of values at t+dt
        k1 = yprime(y0);
        k2 = yprime(y0+0.5*dt*k1);
        k3 = yprime(y0+0.5*dt*k2);
        k4 = yprime(y0+dt*k3);
        y = y0+dt*(k1+2*k2+2*k3+k4)/6;
    end
        
%% FUNCTION FOR THE DIFFERENTIAL EQUATION

    function yp=yprime(y)
    % split out components
        S = y(1);
        E = y(2);
        A = y(3);
        I = y(4);
        J = y(5);
        H = y(6);
        X = fc*c*(I+J)+delta*((1-c)*(I+J)+fa*A);
    % compute derivatives
        Sp = -beta*S*X;
        Ep = -Sp-eta*E;
        Ap = pa*eta*E-alpha*A;
        Ip = (1-pa-ph)*eta*E-gamma*I;
        Jp = ph*eta*E-sigma*J;
        Hp = sigma*J-nu*H;
        Rp = alpha*A+gamma*I+(1-m)*nu*H;
        Dp = m*nu*H;
    % assemble derivative
        yp = [Sp;Ep;Ap;Ip;Jp;Hp;Rp;Dp];
    end

%% END

end

function beta=betaoflambda(lambda,eta,alpha,gamma,sigma,pa,ph,fa)
a = pa*eta./(lambda+alpha);
i = (1-pa-ph)*eta./(lambda+gamma);
j = ph*eta./(lambda+sigma);
x = i+j+fa*a;
beta = (lambda+eta)./x;
end
