function results=Semilin()

% fits data to one or more semilinear models y=Af(x;p)

% each row of results is [pstar,Astar,RSS,AIC]

%   x,y are the lists of data values

%   initial search in [pmin,pmax]
%       search interval shifts if no sign change in slope

%   tol is desired error tolerance for pstar

% uses findmin.m

% by Glenn Ledder 
% written 2021/01/07
% revised 2021/09/07

% direct comments to gledder@unl.edu

%% DATA

% comment out clf to add curve to existing plot

clf

% enter data

x = 0:10:140;
y = [0 7 11 19 19 22 25 21 25 26 23 27 29 30 29];

% specify one to three functions f, g, h

numfncs = 2;

f = @(x,p) x.^p;
g = @(x,p) x./(1+p*x);

% specify initial range of p and tolerance

pmin = [0 0];
pmax = [1 1];
tol = 0.0001;

% choose axis labels

xname = '\it{x}';
yname = '\it{y}';

% choose curve color(s)

col = 'kbg';

%% COMPUTATION

[pstar,Astar,RSS,AIC] = semilinfit(f,pmin(1),pmax(1),tol,x,y);
results = [pstar,Astar,RSS,AIC];

if numfncs>1
    [pstar,Astar,RSS,AIC] = semilinfit(g,pmin(2),pmax(2),tol,x,y);
    results = [results; pstar,Astar,RSS,AIC];
end

if numfncs>2
    [pstar,Astar,RSS,AIC] = semilinfit(h,pmin(3),pmax(3),tol,x,y);
    results = [results; pstar,Astar,RSS,AIC];
end

%% OUTPUT

hold on
box on

% use 'MarkerSize',6 in Octave
plot(x,y,'r.','MarkerSize',12)

xx = linspace(x(1),x(end));

yy = results(1,2)*f(xx,results(1,1));
plot(xx,yy,col(1),'LineWidth',1.4)

if numfncs>1
    yy = results(2,2)*g(xx,results(2,1));
    plot(xx,yy,col(2),'LineWidth',1.4)
end

if numfncs>2
    yy = results(3,2)*h(xx,results(3,1));
    plot(xx,yy,col(3),'LineWidth',1.4)
end

% use 'FontSize',18 in Octave
xlabel(xname,'FontSize',14);
ylabel(yname,'FontSize',14,'Rotation',0);

%% END

end %Semilin


%% FUNCTION semilinfit

function [pstar,Astar,RSS,AIC]=semilinfit(f,pmin,pmax,tol,x,y)

%   finds optimal (p,A) and RSS for the semilinear model y=Af(x;p)
%   f = @(x,p) must be defined in calling program

%   initial search in [pmin,pmax]
%   search interval shifts if no sign change in slope

%   tol is desired error tolerance for pstar

%   x,y are the lists of data values

% define function F(p) in terms of auxiliary function rss
sumy2 = sum(y.^2);
F = @(p) rss(f,p,x,y,sumy2);

% use auxiliary function findmin to find pstar
pstar = findmin(F,pmin,pmax,tol);

% find Astar, RSS, AIC
ff = f(x,pstar);
Astar = sum(y.*ff)/sum(ff.^2);
RSS = rss(f,pstar,x,y,sumy2);
n = length(x);
AIC = n*log(RSS/n)+6;

end %semilinfit


%% FUNCTION rss(f,p,x,y,sumy2)

function z=rss(f,p,x,y,sumy2)
%   computes RSS for the semilinear model y=qf(x;p)
%   x,y are the lists of data values
%   sumy2 is the sum of y^2 values
ff = f(x,p);
z = sumy2-(sum(y.*ff)).^2/sum(ff.^2);
end

