%% SEIR_comparison

% Plots a comparison of population classes for multiple scenarios

% Uses seir.m, 2020/11/27 version

% User specifies a list of values for one of the key parameters:
%   R0 is the basic reproductive number
%   tL is the mean incubation time
%   tI is the mean recovery time
%   i0 is the initial infectious fraction
%   r0 is the initial immune fraction

% e0 is calculated assuming exponential growth phase

% The program is designed so that only two lines need to be modified to
% make a new experiment (see '%%%' comments)
%     line 45 defines the independent variable values
%     line 75 links the independent variable name and values
% The legend is optional

% output figure:
%   left panel: S vs time
%   right panel: I vs time

% by Glenn Ledder
% written 2020/11/27
% revised 2021/03/16

% direct comments to gledder@unl.edu

%% DEFAULT SCENARIO DATA

R0 = 2.5;
tL = 5;
tI = 10;
i0 = 0.001;
r0 = 0;

% use 1 for square axes or 0 for nonfixed aspect ratio

sq = 0;

%% INDEPENDENT VARIABLE DATA

%%% This section needs to be modified for each experiment.

%%% xvals is the set of independent variable values
xvals = [5,4,3,2];

%% COMMON DATA

target = 0.001;
maxdays = 1000;

%% INITIALIZATION

clf
for k=1:2
    subplot(1,2,k)
    hold on
    box on
    if sq>0
        axis square
    end
    xlabel('days')
end

N = length(xvals);
finalS = zeros(1,N);
maxI = zeros(1,N);
maxday = zeros(1,N);

%% COMPUTATION and PLOTS

for n=1:N
    
    %%% The left side of this statement needs to be the independent
    %%% variable for the experiment.
    R0 = xvals(n);

    eta = 1/tL;
    gamma = 1/tI;
    beta = gamma*R0;

    a = eta;
    b = eta-gamma+gamma*R0*i0;
    c = -gamma*R0*(1-i0-r0);
    b2 = b/2;
    rho = (sqrt(b2^2-a*c)-b2)/a;
    e0 = rho*i0;

    [S,~,I,~,sinfty] = seir(R0,tL,tI,e0,i0,r0,target,maxdays);
    days = length(I)-1;
    
    [M,J] = max(I);
    maxI(n) = M;
    maxday(n) = J-1;
    finalS(n) = sinfty;   
    
    subplot(1,2,1)
    plot(0:days,S,'LineWidth',1.4)
    subplot(1,2,2)
    plot(0:days,I,'LineWidth',1.4)
end

subplot(1,2,1)
ylim([0,1])
ylabel('susceptible')
subplot(1,2,2)
ylabel('infectious')
legend('R0=5','R0=4','R0=3','R0=2','Location','Northeast')

%% OUTPUT

maxI = maxI
maxday = maxday
finalS = finalS
