function DiscreteSys()

% runs a simulation for a discrete system x_{t+1}=g(x_t)

% XX is a matrix with columns x1, x2, ..., xn:

% by Glenn Ledder
% written 2021/02/15
% revised 2021/04/02

% direct comments to gledder@unl.edu

%% DATA

% set model parameters

r = 3;
s = 0.6;

% set initial condition and total time

y0 = 2*(r*s-1)/(1+s);
a0 = s*y0;
l0 = y0/r;
X0 = [l0, y0, a0];
T = 50;

% choose which variable(s) to plot

plotvars = [2];

% set scale factor for x1
%   population plot is x1/F rather than x1

F = 1;

% choose name for variable axes

varname1 = '\it{Y}';

% define the function that determines populations

function X1 = newpop(Xt)
    Lt = Xt(1);
    Yt = Xt(2);
    At = Xt(3);
    L1 = At./(1+Yt+At);
    Y1 = r*Lt;
    A1 = s*Yt;
    X1 = [L1 Y1 A1];
end

%% INITIALIZATION

% set up graphs for populations and growth rates

clf
hold on
box on

% set initial condition

X = X0;

% XX is a matrix with the population vectors as columns

sz = length(X0);
XX = zeros(T+1,sz);
XX(1,:) = X0;           % initial condition is row 0

%% COMPUTATION

% compute population values and update data structures

for t=1:T
    X = newpop(X);
    XX(t+1,:) = X;
end

%% OUTPUT

% in Octave, use marker size of 6 for '.'

% plot populations

for j=plotvars
    if j==1
        plot(0:T,XX(:,1)/F,'.','MarkerSize',12)
    else 
        plot(0:T,XX(:,j),'.','MarkerSize',12)
    end
end

% use 'FontSize',18 in Octave
xlabel('\it{t}','FontSize',14);
ylabel(varname1,'FontSize',14,'Rotation',0);

%% END

end

