function N=CobwebPlotter()

% creates a cobweb plot for N_{t+1}=g(N_t)
% also creates a simulation plot if plot2>0

%   X,Y are the lists of data values

% IF USING OCTAVE, MODIFY THE cobweb FUNCTION (AT BOTTOM):
%   cut marker sizes from 6 and 4 to 2
%   change font sizes from 14 to 18

% by Glenn Ledder
% written 2021/01/30
% revised 2021/03/20

% direct comments to gledder@unl.edu

%% DATA

% define model parameters

K = 1000;
R = 1.5;

% define function for N_{t+1}=g(N_t)

g = @(N) N+R*N.*(1-N/K);

% define initial state

N0 = 100;

% choose number of steps

T = 4;

% define plot parameters

maxN = 1600;            % range for variable axis
rows = 1;               % rows in plot structure
cols = 2;               % columns in plot structure
plot1 = 1;              % location number for cobweb plot
plot2 = 2;              % location number for simulation (0 to suppress)
varname = '\it{N}';     % name for variable axis

% use 1 for square axes or 0 for nonfixed aspect ratio

sq = 1;

%% COMPUTATION and OUTPUT

% order cobweb plot

clf
plotdata = [rows cols plot1 plot2 sq];

N = cobweb(g,N0,T,maxN,plotdata,varname);

% add generic y axis label to cobweb plot

% use 'FontSize',18 in Octave
subplot(rows,cols,plot1)
ylabel('\it{y}','FontSize',14,'Rotation',0);

%% END

end


%% FUNCTION cobweb

function N=cobweb(seqfnc,N0,T,maxN,plotdata,varname)

% produces a cobweb plot for N_{t+1}=seqfnc(N_t)

% N0 is initial population
% T is simulation duration
% maxN is axis limits for population axis
% plotdata is a vector: 
%   number of rows
%   number of columns 
%   location number for cobweb plot
%   location number for time history plot or 0 to suppress

% CHANGES FOR OCTAVE
%   cut marker sizes from 6 and 4 to 2
%   change font sizes from 14 to 18

% by Glenn Ledder
% written 2021/01/30
% revised 2021/03/15

% direct comments to gledder@unl.edu

%% Suggested trial values

% seqfnc = @(N) 1.5*N;
% N0 = 1.6;
% T = 3;
% maxN = 6;
% plotdata = [1 2 1 2];
% varname = '\it{N}';

%% Set up the plots

purple = [0.5 0.2 0.5];

rows = plotdata(1);
cols = plotdata(2);
plot1 = plotdata(3);
plot2 = plotdata(4);
sq = plotdata(5);

if plot2>0
    for k=[plot2 plot1]
        subplot(rows,cols,k)
        hold on
        box on
        if sq>0
            axis square
        end
        ylim([0,maxN])
        if k==plot1
            xlim([0,maxN]);
            xlabel(varname,'FontSize',14);
        else
            xlim([0,T]);
            xlabel('\it{t}','FontSize',14);
            ylabel(varname,'Rotation',0,'FontSize',14);
        end
    end
else
    subplot(rows,cols,plot1)
    hold on
    box on
    if sq>0
        axis square
    end
    ylim([0,maxN])
    xlim([0,maxN]);
    xlabel(varname,'FontSize',14);
end

%% Set up the data structures

N = N0*ones(1,T+1);
xx = N0*ones(1,T);
yy = N0*ones(1,T);
webx = N0*ones(1,2*T);
weby = N0*ones(1,2*T);

%% Plot functions

% Plot y=f(N) as a thick blue curve.
% Plot y=N in black.

x = linspace(0,maxN);
f = seqfnc(x);
plot(x,f,'b','LineWidth',1.5)
plot([0 maxN],[0 maxN],'k','LineWidth',1.2)

%% Compute the sequence values by iteration
% (xx,yy) are the points (N_t,N_{t+1})
% (webx,weby) are coordinates for the stairstep cobweb line

for i=1:T
    N(i+1) = seqfnc(N(i));
    xx(i) = N(i);
    yy(i) = N(i+1);
    webx(2*i) = N(i);
    weby(2*i) = N(i+1);
    if i<T
        webx(2*i+1) = N(i+1);
        weby(2*i+1) = N(i+1);
    end
end

%% Mark the points (N_t,N_{t+1}) as black disks.
% Plot cobweb segments in red.

plot(N0,N0,'ks','MarkerSize',6,'MarkerFaceColor','k')
plot(xx,yy,'ko','MarkerSize',4,'MarkerFaceColor','k')
plot(webx,weby,'r')

%% Plot the points (t,N) and dotted lines.

if plot2>0
    
    subplot(rows,cols,plot2)
    plot(0,N0,'ks','MarkerSize',6,'MarkerFaceColor','k')
    plot(1:T,N(2:end),'ko','MarkerSize',4,'MarkerFaceColor','k')

    for t=1:T
        plot([0 t],[N(t+1) N(t+1)],':','Color',purple,'LineWidth',1.5)
    end

    subplot(rows,cols,plot1)
    plot([N0 maxN],[N(1) N(1)],':','Color',purple,'LineWidth',1.5)
    for t=1:T-1
        x1 = max(N(t),N(t+1));
        plot([x1 maxN],[N(t+1) N(t+1)],':','Color',purple,'LineWidth',1.5)
    end
    plot([N(T) maxN],[N(T+1) N(T+1)],':','Color',purple,'LineWidth',1.5)

end

%% END

end

