%
% This is a demo/tutorial for the best-fit search program 'odelinesearch.m'
% The most import input to the program includes three items: the model, the
% data, and the fit function of the model output and the data. The program
% finds a best-fit parameter of the model to the data, with the fit error
% being the per-data-point relative Euclidean error. The example used for
% the demo is to fit an excitable membrane model to Hodgkin and Huxley's
% squid giant axon 17 from their seminal paper published in 1952. 
%

clear all
%
%  User Defined Input of the program
%
ModelName = 'Dengmodel'; %
options=odeset('e','off','BDF','on','AbsTol',1e-6,'RelTol',1e-6); % for Matlab ode solver, use =[] for default.

VariableNames = {'V' 'G_K' 'G_Na' 'G_G'};
ParameterNames = {'E_K','barg_K','b_K', 'E_Na','barg_Na','b_Na', 'E_G','barg_G','b_G', 'C', 'Iin', 't_depol', 'timescale', 'tau_K', 'tau_Na', 'tau_G'}; %

%%
%%% Specify searching parameters. 
%%%

%Parameters4Search={'E_K','barg_K','b_K', 'E_Na','barg_Na','b_Na', 'E_G','barg_G','b_G', 'C', 'Iin', 't_depol', 'timescale', 'tau_K', 'tau_Na', 'tau_G'}; %
Parameters4Search={               'b_K',        'barg_Na','b_Na', 'E_G','barg_G','b_G',                                      'tau_K', 'tau_Na'          }; %

%Variables4Search={'G_K','G_Na'}; 
Variables4Search={};

%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%% Input Control Parameters %%%%%%%%%%%
RunOrNoRun=1; % Binary. If 0, no search is done but output the initial fit. If =1, search for each of the InitialConditions

NumberOfRun=1; % if RunOrNoRun=1, NumberOfRun are searched with InitialConditions and the rest randomly choosen within the RangeScale of the first of InitialConditions.

RangeScale=1.2; % Range scale of the parameter when multiple random searches are carried out within the range defined by it.

NumberOfSearchPartition=15; % ... on each search line in one direction of the initial/iterative fit.

%transienttime=0; % to set an initial state before data is fitted to the fit functional.

%%%% Specify the stopage conditions for the search loops
StopThreshold=.0001;
MaximalSearchIterate=inf; % A finite number if StopThreshold is not met, otherwise inf

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%% Input initial fit variables and parameters %%%%%%%%%%%%%%%%%%%%

E_K=-62+2.5; % HH value
barg_K=0.0229; % HH value
b_K=9.0; %offset 1  

E_Na=65+2.5; %HH value
barg_Na=100;%barg_Na=90;
b_Na=9; %offset 1 

E_G=-55+2.5;
barg_G=10;
b_G=7; %offset 1 

C=.5; %offset .5; HH value

Iin=0; % =0 for no external current.
t_depol=.15; timescale=1;
            
tau_K=1;tau_Na=10; tau_G=10;

param=[E_K, barg_K, b_K, E_Na, barg_Na, b_Na, E_G, barg_G, b_G, C, Iin, t_depol, timescale, tau_K, tau_Na, tau_G];
       
xinit=[-23.1707+2.5   exp((-50+2.5-E_K)./(b_K+1))   exp((-50+2.5-E_Na)./(b_Na+1))  exp(-(-50+2.5-E_G)./(b_G+1))]; %

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

InitialConditions=[xinit param]; % 

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% or if RunOrNoRun> length of InitialConditions, random searches are
% included. 
%
pstvsign=(InitialConditions(1,:)>0);
ngtvsign=1-pstvsign;
InitialSearchRangeLow=(1/RangeScale)*pstvsign.*InitialConditions(1,:)+RangeScale*ngtvsign.*InitialConditions(1,:);
InitialSearchRangeHigh=RangeScale*pstvsign.*InitialConditions(1,:)+(1/RangeScale)*ngtvsign.*InitialConditions(1,:);

if RunOrNoRun>0
    NumberOfVariables=length(VariableNames); 
    NumberOfParameters=length(ParameterNames);
    VariableNdParameterNames= union(VariableNames,ParameterNames,'stable');
    VariablesNdParameters4Search = union(Variables4Search,Parameters4Search,'stable');

    SearchVariablesNdParameters=zeros(1,NumberOfVariables+NumberOfParameters); %creat binary sequence with 1 for search 0 for nonsearch.
    for k = 1:length(VariablesNdParameters4Search)
        jj = strcmp(VariablesNdParameters4Search(k),VariableNdParameterNames);
        SearchVariablesNdParameters = SearchVariablesNdParameters+jj;
    end
        rand(ceil(50*rand)); 
        InitialSearchPoint=rand(NumberOfRun, NumberOfVariables+NumberOfParameters);
        InitialSearchPoint=ones(NumberOfRun,1)*InitialSearchRangeLow+InitialSearchPoint.*(ones(NumberOfRun,1)*(InitialSearchRangeHigh-InitialSearchRangeLow));
            SearchVariablesNdParametersArray = ones(NumberOfRun, 1)*SearchVariablesNdParameters;
            NonSearchVariablesNdParametersArray = (1-SearchVariablesNdParametersArray);
    InitialSearchPoint =InitialSearchPoint.*SearchVariablesNdParametersArray + NonSearchVariablesNdParametersArray.*(ones(NumberOfRun, 1)*InitialConditions(1,:));
    InitialSearchPoint(1:length(InitialConditions(:,1)),:)= InitialConditions;
    InitialConditions=InitialSearchPoint;
    %OriginalSearchPoint=InitialSearchPoint;
end
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% Data Structure for User Defined Input %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Data=[allinclusivetime,data1,data1timeweight1,...,datan,datantimeweightn]
% dateweight at row i is >0 but <=1 for observed at time t_i with given weight, 0 otherwise. 
% 
HH52eAxon17 =[0   26.8293
    0.0735   20.7317
    0.1224   18.2927
    0.2204   17.0732
    0.3184   18.2927
    0.4163   19.5122
    0.5388   23.1707
    0.6367   30.4878
    0.7592   45.1220
    0.8816   75.6098
    0.9306   96.3415
    0.9796  102.4390
    1.0531  106.0976
    1.1020  107.3171
    1.2490  104.8780
    1.4694   95.1220
    1.9347   64.6341
    2.4735   24.3902
    2.8163    2.4390
    3.0367   -3.6585
    3.2571   -7.3171
    3.6490   -8.5366
    4.6286   -7.3171
    5.0694   -7.8171
    6.0000   -4.8780]; % HH Axon 17 data from [HH1952]
%%

ExperimentData(:,1:2) =[HH52eAxon17(:,1), HH52eAxon17(:,2)-50+2.5]; % 

    DataWeightTestOrder =[1 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0]';    
    DataWeightFirstOrder=[1 0 0 1 0 0 0 0 0 0 0 0 0 1 0 0 0 0 0 0 0 1 0 0 1]'; % first order fit
    DataWeightSecondOrder=[1 0 0 1 0 0 0 0 0 1 0 0 0 1 0 0 1 0 0 0 1 1 1 0 1]'; % second order fit 
    DataWeightAll=[1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1 1]'; % all fit

    DataWeight=DataWeightSecondOrder;

ExperimentData(:,3)=DataWeight;

% Start potasium current estimate
ExperimentData(:,4:5)=0*HH52eAxon17;
% End potasium current estimate

% Start sodium current estimate
ExperimentData(:,6:7)=0*HH52eAxon17;
% End sodium current estimate

% Start gating current estimate
ExperimentData(:,8:9)=0*HH52eAxon17;
% End gating current estimate

ExperimentData(:,5)=0; % 0 for not using current for fit.
ExperimentData(:,7)=0; % 0 for not using current for fit.
ExperimentData(:,9)=0; % 0 for not using current for fit.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% End Experimental Data Input
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% Fit functional input %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%%%%%%%%%%%%%%
%%%%%%%%%%%%%% PredictedState is a user defined functional which is used to
%%%%%%%%%%%%%% fit the ExperimentalData. Its variables are 'time,
%%%%%%%%%%%%%% variables, parameters' in column, its output are predicted
%%%%%%%%%%%%%% states consistent with the ExperimentData in its dimensional
%%%%%%%%%%%%%% culumns, excluding the time and the weights columns. 
%%%%%%%%%%%%%%

PredictedState=@fitfunction;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%% End of All Program Input.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%
%
% Run raw search and save the data. 
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%% Structure of Output
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%% BestFit=zeros(numbrofrun,(NumberOfVariables+NumberOfParameters) + 1 + (NumberOfVariables+NumberOfParameters) + (NumberOfVariables+NumberOfParameters) + 4); 
% the first set is for the Values, the middle 1 is for the Error, 
% the second is for the Sensitivity of best fit, the last is for the
% initial fit, and 4 more columns record
% the search parameters: StopThreshold, MaximalSearchIterate,
% NumberOfSearchPartition, transienttime.
% Sorted in the increasing value of the error. 

%%%% Errorfunction=zeros(2*NumberOfSearchPartition+1,2,NumberOfVariables+NumberOfParameters, numbrofrun)-1;
%
% column 1 for searched points of one parameter line, column 2 for 
% the corresponding error. Page dimension for NumberOfVarNdParm. Page2
% dimension is for the number of run
%

%%%% StateFit2Data=zeros(length(ExperimentData(:,1)),NumberOfVariables+1,numbrofrun);
%
% Same row length as Experiment Data, columns for state variables plus time, page for
% numbrofrun.
%
NumberOfVariables=length(VariableNames); 
NumberOfParameters=length(ParameterNames);

tic
[FitInit, FitParam, FitError, FitSens, StateFit2Data, Errorfunction, SearchStepTaken]=odelinesearch(ModelName,options,VariableNames, ParameterNames, Variables4Search, ...
    Parameters4Search, ExperimentData, PredictedState, InitialConditions, RunOrNoRun, NumberOfSearchPartition, StopThreshold, MaximalSearchIterate);
toc


%%
% Save output to a file. 
%save BestFitHHAxon17DengModel.mat ModelName options VariableNames ParameterNames Variables4Search Parameters4Search ExperimentData PredictedState FitInit FitParam FitError FitSens StateFit2Data Errorfunction SearchStepTaken  RunOrNoRun NumberOfRun NumberOfSearchPartition StopThreshold MaximalSearchIterate InitialConditions RangeScale 

%InitialConditions=[FitInit(k,:), FitParam(k,:)];
%save BestInitialsDengmodel.mat InitialConditions

%%

return
